//! A basic example showing how to build a simple app using the
//! Rust DittoSDK.
//!
//! This exmaple will create a basic peer and register a single subscription
//! with a central provider (tcp+tls://hydra.dev.ditto.io:34468/).
//!
//! It can then be used to generate a stream of events for purposes of measuring
//! bounce latency
//!
//! Example Invocation:
//!
//!  ``` shell
//!  RUST_LOG=debug DITTO_APP_NAME=live.ditto.carsapp DITTO_SITE_ID=1 \
//!  DITTO_DB_PATH=./sender cargo run -p dittokit --example bounce_latency_tx
//!  ```
//! Note that it is vital you clear the local database files (`rm -rf sender` in
//! the example above) between runs in order to keep the replication state
//! intact
//!
//! Currently assumes HyDRA subscription server is listening on localhost:4040.
//! This can be arranged with a k8s cluster by running
//! ```shell
//! kubectl port-forward <sub-server pod> 4040
//! ```

extern crate safer_ffi;
extern crate tempdir;

use std::{env, path::PathBuf};

use chrono::Utc;
use dittolive_ditto::prelude::*;
use rand::Rng;
use serde::{Deserialize, Serialize};

#[derive(Serialize, Deserialize, Debug)]
struct LatencyProbe {
    timestamp: i64,
    subsec_nanos: u32,
}

fn main() {
    env_logger::init();
    // All Ditto Apps require the following:
    // - a unique AppName, following domain name syntax
    // - a directory for data storage
    // - at least one ordered, TCP-like socket for sending bytes
    // - a SiteId, ActorId which uniquely identifies this peer across restarts
    // - a license token
    // - a bundle of PKI certificates consisting of a private key (for this device),
    //   an identity certificate (indicating the tenant), and a CA Cert for
    //   authenticating the identity certificates of peers
    //
    // For now your app name must be "live.ditto.carsapp"
    let app_name =
        env::var("DITTO_APP_NAME").unwrap_or_else(|_x| String::from("live.ditto.carsapp"));
    // define actor/site ID
    // If absent a random one will be created
    let site_id: u64 = env::var("DITTO_SITE_ID")
        .map(|x| x.parse::<u64>().unwrap())
        .unwrap_or_else(|_x| {
            // recall the site ID's 0 and 1 have special meaning
            ::rand::thread_rng().gen_range(2..SiteId::MAX)
        });
    let license_token: String = env::var("DITTO_LICENSE").unwrap();
    let data_dir = env::var_os("DITTO_DB_PATH")
        .map(PathBuf::from)
        .or_else(|| -> Option<PathBuf> { Some(std::env::temp_dir()) })
        .map(|mut path: PathBuf| {
            path.push("dittokit");
            let _ = std::fs::create_dir_all(&path);
            path
        });

    let hydra_host: String =
        env::var("HYDRA_HOST").unwrap_or_else(|_| "localhost:4040".to_string());

    let identity = Identity::new_development(&app_name, site_id, data_dir.as_deref()).unwrap();
    let mut ditto = DittoKit::new(identity, data_dir);
    ditto.set_access_license(license_token.as_str());

    let mut transport_config = TransportConfig::new();
    transport_config.connect.tcp_servers.insert(hydra_host);
    ditto.set_transport_config(transport_config);
    ditto.start_sync();

    // This block can be used to set up a subscription and live query for local and
    // remote documents. The Documents and LiveQueryEvents processed by an
    // EventHandler callback, which can use a channel to push them back to the
    // main thread for printing.
    let store = ditto.store();
    let collection = store.collection("testcollection");
    let (tx, rx) = std::sync::mpsc::sync_channel(120);
    let event_handler = move |documents: Vec<ffi_sdk::BoxedDocument>, event| {
        // Remember these are generally LOCAL document updates
        ::log::trace!(
            "Latency Sender received event {:?} with {} modified documents",
            &event,
            documents.len()
        );
        for doc in documents.into_iter() {
            let _ = tx.send(doc);
        }
    };

    // This loop inserts new docs into the LOCAL collection
    // It uses a background thread so the main thread is free to respond
    // LiveQueryEvents
    std::thread::spawn(move || loop {
        let store = ditto.store();
        let collection = store.collection("testcollection");
        let now = Utc::now();
        let msg = LatencyProbe {
            timestamp: now.timestamp(),
            subsec_nanos: now.timestamp_subsec_nanos(),
        };
        let id = DocumentId::new(&now.to_rfc3339()).unwrap();
        let _doc_id = collection.unwrap().insert(msg, Some(&id), false);
        std::thread::sleep(std::time::Duration::from_millis(1000));
    });

    // this registers and starts the live query
    // this returns IMMEDIATELY and will run until the LiveQuery (Handle) is dropped
    // Note that Find ALL is not presently supported by the Subscription Server
    let start_time = Utc::now().timestamp(); // The time of this session
                                             // this will filter for only events in the current test session
    let query: String = format!("timestamp >= {}", start_time);
    let _lq = collection.unwrap().find(&query).observe(event_handler);

    for doc in rx.iter() {
        ::log::trace!("Latency Sender inserted document {:?} ", &doc);
    }
}
