//! A basic example showing how to build a simple app using the
//! Rust DittoSDK.
//!
//! This exmaple will create a basic peer and register a single subscription
//! with a central provider (tcp+tls://hydra.dev.ditto.io:34468/).
//!
//! It can then be used to generate a stream of events for purposes of measuring
//! bounce latency

extern crate safer_ffi;

use std::env;

use chrono::Utc;
use dittolive_ditto::prelude::*;
use serde::{Deserialize, Serialize};

#[derive(Serialize, Deserialize, Debug)]
struct LatencyProbe {
    timestamp: i64,
    subsec_nanos: u32,
}

fn main() -> Result<(), Box<dyn std::error::Error>> {
    env_logger::init();
    let mut ditto = Ditto::builder()
        .with_temp_dir()
        .with_minimum_log_level(CLogLevel::Info)
        .with_identity(|ditto_root| {
            let app_id = AppId::from_env("BOUNCE_LATENCY_APP_ID")?; // MUST match that of the Receiver App

            // Override this auth url outside of prod
            let auth_url =
                std::env::var("DITTO_AUTH_URL").unwrap_or_else(|_x| app_id.default_auth_url());
            let auth_event_handler = MyHandler::default();
            OnlineWithAuthentication::new(
                ditto_root,
                app_id,
                auth_event_handler,
                false,
                Some(&auth_url),
            )
        })?
        .with_transport_config(|_identity| {
            let mut config = TransportConfig::new();
            if let Ok(sync_url) = env::var("DITTO_SYNC_URL") {
                config.connect.websocket_urls.insert(sync_url);
            }
            config
        })?
        .build()?;
    ditto.set_license_from_env("DITTO_LICENSE")?;
    ditto.try_start_sync()?;

    // This block can be used to set up a subscription and live query for local and
    // remote documents. The Documents and LiveQueryEvents processed by an
    // EventHandler callback, which can use a channel to push them back to the
    // main thread for printing.
    let store = ditto.store();
    let collection = store.collection("testcollection");
    let (tx, rx) = std::sync::mpsc::sync_channel(120);
    let event_handler = move |documents: Vec<ffi_sdk::BoxedDocument>, event| {
        // Remember these are generally LOCAL document updates
        ::log::trace!(
            "Latency Sender received event {:?} with {} modified documents",
            &event,
            documents.len()
        );
        for doc in documents.into_iter() {
            let _ = tx.send(doc);
        }
    };

    // This loop inserts new docs into the LOCAL collection
    // It uses a background thread so the main thread is free to respond
    // LiveQueryEvents
    std::thread::spawn(move || loop {
        let store = ditto.store();
        let collection = store.collection("testcollection");
        let now = Utc::now();
        let msg = LatencyProbe {
            timestamp: now.timestamp(),
            subsec_nanos: now.timestamp_subsec_nanos(),
        };
        let id = DocumentId::new(&now.to_rfc3339()).unwrap();
        let _doc_id = collection.unwrap().insert(msg, Some(&id), false);
        std::thread::sleep(std::time::Duration::from_millis(1000));
    });

    // this registers and starts the live query
    // this returns IMMEDIATELY and will run until the LiveQuery (Handle) is dropped
    // Note that Find ALL is not presently supported by the Subscription Server
    let start_time = Utc::now().timestamp(); // The time of this session
                                             // this will filter for only events in the current test session
    let query: String = format!("timestamp >= {}", start_time);
    let _lq = collection.unwrap().find(&query).observe(event_handler);

    for doc in rx.iter() {
        ::log::trace!("Latency Sender inserted document {:?} ", &doc);
    }
    Ok(())
}

struct MyHandler {
    token: String,
    provider: String,
}

impl DittoAuthenticationEventHandler for MyHandler {
    fn authentication_required(&self, auth: dittolive_ditto::auth::DittoAuthenticator) {
        if let Err(e) = auth.login_with_token(&self.token, &self.provider) {
            ::log::error!("Login Failed {}", &e);
        }
    }

    fn authentication_expiring_soon(
        &self,
        _auth: dittolive_ditto::auth::DittoAuthenticator,
        seconds_remaining: std::time::Duration,
    ) {
        ::log::info!(
            "Auth token expiring in {} seconds",
            seconds_remaining.as_secs()
        );
    }
}

impl Default for MyHandler {
    fn default() -> Self {
        MyHandler {
            token: String::from("letmein"),
            provider: String::from("test-provider"),
        }
    }
}
