use_prelude!();

use std::sync::Arc;

use crate::error::{DittoError, ErrorKind};

/// A handle for a Ditto Subscription to documents in a collection matching a
/// query.
///
/// When the Subscription is dropped, it will automatically trigger a clean up
/// of itself.
pub struct Subscription {
    pub(super) ditto: Arc<ffi_sdk::BoxedDitto>,
    pub(super) collection_name: char_p::Box,
    pub(super) query: char_p::Box,
    pub(super) query_args: Option<Vec<u8>>,
}

impl Subscription {
    pub fn new(
        ditto: Arc<ffi_sdk::BoxedDitto>,
        collection_name: char_p::Box,
        query: &str,
        query_args: Option<Vec<u8>>,
    ) -> Self {
        let query = char_p::new(query);
        unsafe {
            let code = ffi_sdk::ditto_add_subscription(
                &ditto,
                collection_name.as_ref(),
                query.as_ref(),
                query_args.as_ref().map(|qa| (&qa[..]).into()),
            );
            if code != 0 {
                ::log::error!(
                    "Error adding subscription for collection {} query {}. Error: {}",
                    &collection_name,
                    &query,
                    DittoError::from_ffi(ErrorKind::InvalidInput)
                );
            }
        };
        Subscription {
            ditto,
            collection_name,
            query,
            query_args,
        }
    }
}

impl Drop for Subscription {
    fn drop(&mut self) {
        ::log::debug!(
            "Dropping subscription for collection {} query {}",
            &self.collection_name,
            &self.query
        );
        unsafe {
            let code = ffi_sdk::ditto_remove_subscription(
                &self.ditto,
                self.collection_name.as_ref(),
                self.query.as_ref(),
                self.query_args.as_ref().map(|qa| (&qa[..]).into()),
            );
            if code != 0 {
                ::log::error!(
                    "Error removing subscription for collection {} query {}. Error: {}",
                    &self.collection_name,
                    &self.query,
                    DittoError::from_ffi(ErrorKind::InvalidInput)
                );
            }
        }
    }
}
