//! Example Ditto Cars App "Relay"
//!
//! Listens for incoming connections from the Ditto Car's App
//! on BLE, LAN/Multicast, and TCP transports
//! syncing data with connected peers.
//!
//! Example Invocation
//! ``` shell
//! cargo run -p dittolive-ditto --example carsapp
//! ```

extern crate dittolive_ditto;
extern crate log;
extern crate safer_ffi;
extern crate serde;
extern crate serde_cbor;

use std::str::FromStr;

use dittolive_ditto::prelude::*;
fn main() -> Result<(), Box<dyn std::error::Error>> {
    env_logger::init();
    let mut ditto = Ditto::builder()
        .with_temp_dir()
        .with_identity(|ditto_root| {
            // expected by other apps
            let app_id = AppId::from_str("d0e9a40e-9beb-4c90-bb38-769269cbb6fb")?;
            // We don't want a fully random Development Identity as we need to make sure we
            // have a specific AppId shared by all peers
            identity::Development::new(ditto_root, app_id)
        })?
        .with_minimum_log_level(CLogLevel::Info)
        .with_transport_config(|_identity| -> TransportConfig {
            let mut transport_config = TransportConfig::new();
            transport_config.listen.tcp.enabled = true;
            transport_config.listen.tcp.interface_ip = "0.0.0.0".to_string();
            transport_config.listen.tcp.port = 4040;
            transport_config.peer_to_peer.bluetooth_le.enabled = true;
            transport_config.peer_to_peer.lan.enabled = true;
            transport_config
        })?
        .build()?;
    ditto.set_license_from_env("DITTO_LICENSE")?;
    ditto.try_start_sync()?;
    ::log::debug!("Ditto Cars App started!");

    let store = ditto.store();
    let collection = store.collection("cars");

    let (tx, rx) = std::sync::mpsc::sync_channel(120);
    // This handler is called every time docs from local or remote sources are
    // committed to the local store which match the associated query.
    // `documents` is a vec of ALL documents matching the query after application of
    // the transaction.
    // `event` can be used to dissect out which of these are insertions.
    let event_handler = move |documents: Vec<ffi_sdk::BoxedDocument>, event| {
        ::log::trace!(
            "Latency Receiver got {:?} with {} updated documents",
            &event,
            documents.len()
        );
        match event {
            LiveQueryEvent::Initial { .. } => {
                // On an initial sync, we can calculate the latency for arrival of the first
                // document
                ::log::info!("Initial Data loaded");
            }
            LiveQueryEvent::Update { insertions, .. } => {
                // We only want to send the newest event
                for idx in insertions.iter() {
                    if let Some(car) = documents.get(*idx) {
                        let car_cbor = car.to_cbor().unwrap();
                        let _ = tx.send(car_cbor);
                    }
                }
            }
        }
    };
    // downgrade our logging output before running the query
    Ditto::set_minimum_log_level(CLogLevel::Debug);
    // Find and report on all cars
    let _lq = collection.unwrap().find_all().observe(event_handler);
    for car_cbor in rx.iter() {
        let car_str = serde_json::to_string(&car_cbor).unwrap();
        println!("{}", &car_str);
    }
    Ok(())
}
