use std::{fs::File, io::Read, path::Path, time::Duration};

pub use ::safer_ffi::prelude::*;
use chrono::{DateTime, TimeZone, Utc};
pub use dittolive_ditto::{auth::DittoAuthenticator, error::DittoError, identity, prelude::*};
use serde::{Deserialize, Serialize};

#[allow(dead_code)]
pub fn test_license_from_file<P: AsRef<Path>>(path: P) -> std::io::Result<String> {
    let mut f = File::open(path)?;
    let mut key = String::new();
    let _bytes = f.read_to_string(&mut key)?;
    let result = key.trim().to_string();
    if result.is_empty() {
        Err(std::io::Error::new(
            std::io::ErrorKind::InvalidData,
            "License Key is empty",
        ))
    } else {
        Ok(result)
    }
}

#[allow(dead_code)]
pub fn get_ditto() -> Result<Ditto, DittoError> {
    let mut ditto = Ditto::builder()
        .with_temp_dir()
        .with_identity(|ditto_root| identity::Development::random(ditto_root))?
        .with_minimum_log_level(CLogLevel::Info)
        .build()?;
    ditto.set_license_from_env("DITTO_LICENSE")?;
    Ok(ditto)
}

#[allow(dead_code)]
pub fn get_inactive_ditto() -> Result<Ditto, DittoError> {
    let ditto = Ditto::builder()
        .with_temp_dir()
        .with_identity(|ditto_root| identity::Development::random(ditto_root))?
        .with_minimum_log_level(CLogLevel::Info)
        .build()?;
    Ok(ditto)
}

// A Re-usable test document type
#[derive(Serialize, Deserialize, Debug, PartialEq, Eq, Clone)]
pub enum TestColor {
    Red,
    Blue,
    Crimson,
    Orange,
    Yellow,
    Pink,
    White,
    Black,
    Brown,
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Eq, Clone)]
pub struct TestType {
    pub make: String,
    pub color: TestColor,
    pub age: u8,
    pub mileage: i64,
    pub purchased_on: DateTime<Utc>,
    pub tags: Vec<String>,
    pub data: Vec<u8>, // most common repr for bin data in std lib
}

impl TestType {
    #[allow(dead_code)] // This actually is used
    pub fn random() -> TestType {
        use ::rand::rngs::StdRng;
        use rand::{Fill, SeedableRng};

        let mut bytes: Vec<u8> = Vec::with_capacity(2048);
        let mut rng = StdRng::seed_from_u64(123456789);
        bytes.try_fill(&mut rng).unwrap();

        TestType {
            make: String::from("Honda"),
            color: TestColor::Red,
            age: 12,
            mileage: 52000,
            purchased_on: Utc::now(),
            tags: vec![
                String::from("a"),
                String::from("-1"),
                String::from("test_tag"),
            ],
            data: bytes,
        }
    }
}

impl Default for TestType {
    fn default() -> TestType {
        TestType {
            make: String::default(),
            color: TestColor::Red,
            age: 0,
            mileage: 0,
            purchased_on: Utc.ymd(1970, 1, 1).and_hms(0, 0, 0),
            tags: Vec::new(),
            data: Vec::new(),
        }
    }
}

pub struct LoginSource {
    /// The token used for Auth
    pub token: String,
    /// The name of the Authentication Provider configured for this App in Ditto
    /// Cloud
    pub provider: String,
}

impl DittoAuthenticationEventHandler for LoginSource {
    fn authentication_required(&self, auth: DittoAuthenticator) {
        auth.login_with_token(&self.token, &self.provider).unwrap()
    }

    fn authentication_expiring_soon(&self, _auth: DittoAuthenticator, seconds_remaining: Duration) {
        log::debug!("Login will expire in {:?} seconds", seconds_remaining);
    }
}
