use std::sync::{Arc, RwLock};

use ffi_sdk::BoxedAuthClient;

use crate::{
    auth::{DittoAuthenticationEventHandler, DittoAuthenticator, LoginProvider, ValidityListener},
    ditto::AppId,
    error::{DittoError, ErrorKind},
    transport::Transports,
};

use_prelude!();

// Identity is pub because it is part of the Builder's public interface
// It is therefore Sealed to prevent downstream implementations
pub trait Identity: private::Sealed {
    /// Returns the current SiteId identifying the Ditto peer
    fn site_id(&self) -> SiteId;

    /// Returns a shared reference to the underlying AuthClient
    fn auth_client(&self) -> Arc<BoxedAuthClient>;

    /// Returns the underlying `DittoAuthenticator` if specified
    fn authenticator(&self) -> Option<DittoAuthenticator>;

    /// Constructs a `ValidityListener` given a shared reference to the Ditto
    /// `Transports`
    fn make_listener(&self, transports: Arc<RwLock<Transports>>) -> Option<Arc<ValidityListener>>;

    /// Returns the current AppId
    fn app_id(&self) -> Option<&AppId>;

    /// Returns the current DittoRoot
    fn ditto_root(&self) -> Arc<dyn DittoRoot>;

    /// Returns if the curernt web auth token is valid
    fn is_web_valid(&self) -> bool;

    /// Returns if the configured x509 certificate is valid
    fn is_x509_valid(&self) -> bool;

    /// Indicates if cloud sync should be enabled by default
    fn is_cloud_sync_enabled(&self) -> bool;

    /// Returns the configured URL for Auth
    fn auth_url(&self) -> Result<String, DittoError>;

    /// Returns the configured URL for websocket sync
    fn sync_url(&self) -> Result<String, DittoError>;
}

/// Run Ditto in secure production mode, logging on to Ditto Cloud or an
/// on-premises authentication server. User permissions are centrally managed.
/// Sync will not work until a  successful login has occurred.
///
/// The only required configuration is the application's UUID, which can be
/// found on the Ditto portal where the app is registered.
///
/// By default cloud sync is enabled.
/// This means the SDK will sync to a Big Peer in Ditto's cloud when an internet
/// connection is available. This is controlled by the `enableDittoCloudSync`
/// parameter. If `true` (default), a suitable wss:// URL will be added to the
/// `TransportConfig`.
///
/// To prevent cloud sync, or to specify your own URL later, pass `false`.
///
/// Authentication requests are handled by Ditto's cloud by default, configured in the portal at * https://portal.ditto.live.
///
/// To use a different or on-premises authentication service, pass a custom
/// HTTPS base URL as the * `auth_url` parameter.
pub struct Online {
    ditto_root: Arc<dyn DittoRoot>,
    auth: DittoAuthenticator,
    app_id: AppId,
    enable_cloud_sync: bool,
    auth_url: String,
}

/// Test a Ditto Cloud app without authentication ("Playground mode").
/// This mode offers no security and must only be used for development.
/// Other behavior mirrors the `Online` identity.
pub struct OnlinePlayground {
    ditto_root: Arc<dyn DittoRoot>,
    auth: DittoAuthenticator,
    app_id: AppId,
    enable_cloud_sync: bool,
    auth_url: String,
}

/// An identity where any device is trusted provided they know the secret key.
/// This is a simplistic authentication model normally only suitable for private
/// apps where users and devices are both trusted. In this mode, any string may
/// be used as the app id.
pub struct SharedKey {
    ditto_root: Arc<dyn DittoRoot>,
    app_id: AppId,
    auth: DittoAuthenticator,
}

/// An identity where devices are manually configured with a x509 certificate
/// bundle
pub struct Manual {
    ditto_root: Arc<dyn DittoRoot>,
    auth: DittoAuthenticator,
}

/// Develop peer-to-peer apps with no cloud connection. This mode offers no
/// security and must only be used for development. In this mode, any string can
/// be used as the name of the app.
pub struct Development {
    ditto_root: Arc<dyn DittoRoot>,
    app_id: AppId,
    auth: DittoAuthenticator,
}

impl Online {
    /// Construct an Online Identity
    /// * `ditto_root` - The directory containing ditto local storage
    /// * `app_id` - The unique identifier of the application, usually a UUIDv4
    ///   string
    /// * `auth_event_handler` - A user created type which implements the
    ///   `DittoAuthenticationEventHandler` trait
    /// * `enable_cloud_sync` - where cloud sync should be started by default
    /// * `custom_auth_url` - an optional custom authentication URL; default is
    ///   is `cloud.ditto.live`
    pub fn new(
        ditto_root: Arc<dyn DittoRoot>,
        app_id: AppId,
        auth_event_handler: impl DittoAuthenticationEventHandler + 'static,
        enable_cloud_sync: bool,
        custom_auth_url: Option<&str>,
    ) -> Result<Self, DittoError> {
        let auth_url = match custom_auth_url {
            Some(url) => url.to_owned(),
            None => app_id.default_auth_url(),
        };
        let c_working_dir = ditto_root.data_dir_to_c_str()?;
        let c_app_id = char_p::new(app_id.to_string());
        let c_auth_url = char_p::new(auth_url.as_str());

        let login_provider = LoginProvider::new(auth_event_handler);

        let LoginProvider {
            _provider: c_provider,
            ctx,
        } = login_provider;

        let auth_client = unsafe {
            ffi_sdk::ditto_auth_client_make_with_web(
                c_working_dir.as_ref(),
                c_app_id.as_ref(),
                c_auth_url.as_ref(),
                Some(c_provider),
            )
            .ok_or(ErrorKind::Config)?
        };

        let authenticator = DittoAuthenticator::new(Arc::new(auth_client));
        ctx.lock().unwrap().set_authenticator(authenticator.clone());

        Ok(Online {
            app_id,
            ditto_root,
            auth: authenticator,
            enable_cloud_sync,
            auth_url,
        })
    }
}

impl Identity for Online {
    fn site_id(&self) -> SiteId {
        unsafe { ffi_sdk::ditto_auth_client_get_site_id(&self.auth_client()) }
    }

    fn app_id(&self) -> Option<&AppId> {
        Some(&self.app_id)
    }

    fn auth_client(&self) -> Arc<BoxedAuthClient> {
        self.auth.auth_client()
    }

    fn ditto_root(&self) -> Arc<dyn DittoRoot> {
        self.ditto_root.clone()
    }

    fn is_web_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_web_valid(&self.auth_client()) != 0 }
    }

    fn is_x509_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_x509_valid(&self.auth_client()) != 0 }
    }

    fn is_cloud_sync_enabled(&self) -> bool {
        self.enable_cloud_sync
    }

    fn auth_url(&self) -> Result<String, DittoError> {
        Ok(self.auth_url.to_owned())
    }

    fn sync_url(&self) -> Result<String, DittoError> {
        Ok(self.app_id.default_sync_url())
    }

    fn authenticator(&self) -> Option<DittoAuthenticator> {
        Some(self.auth.clone())
    }

    fn make_listener(&self, transports: Arc<RwLock<Transports>>) -> Option<Arc<ValidityListener>> {
        Some(ValidityListener::new(transports, self.auth.auth_client()))
    }
}

impl Development {
    /// Construct a `Development` Identity
    /// * `ditto_root` - Location for the local database
    /// * `app_id` - A unique string identifying the App. For a Development
    ///   Identity this does not need to be a UUIDv4.
    pub fn new(ditto_root: Arc<dyn DittoRoot>, app_id: AppId) -> Result<Self, DittoError> {
        let c_data_dir = ditto_root.data_dir_to_c_str()?;
        let c_app_id = app_id.to_c_string();
        let auth_client = unsafe {
            ffi_sdk::ditto_auth_client_make_for_development(
                Some(c_data_dir.as_ref()),
                c_app_id.as_ref(),
                0,
            )
            .ok_or(ErrorKind::Config)?
        };
        let authenticator = DittoAuthenticator::new(Arc::new(auth_client));
        Ok(Self {
            ditto_root,
            app_id,
            auth: authenticator,
        })
    }

    /// Generate a Development identity with a random AppId for testing
    pub fn random(ditto_root: Arc<dyn DittoRoot>) -> Result<Self, DittoError> {
        let app_id = AppId::generate();
        Self::new(ditto_root, app_id)
    }
}

impl Identity for Development {
    fn site_id(&self) -> SiteId {
        unsafe { ffi_sdk::ditto_auth_client_get_site_id(&self.auth_client()) }
    }

    fn auth_client(&self) -> Arc<BoxedAuthClient> {
        self.auth.auth_client()
    }

    fn app_id(&self) -> Option<&AppId> {
        Some(&self.app_id)
    }

    fn ditto_root(&self) -> Arc<dyn DittoRoot> {
        self.ditto_root.clone()
    }

    fn is_web_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_web_valid(&self.auth_client()) != 0 }
    }

    fn is_x509_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_x509_valid(&self.auth_client()) != 0 }
    }

    fn is_cloud_sync_enabled(&self) -> bool {
        false
    }

    fn auth_url(&self) -> Result<String, DittoError> {
        Err(DittoError::new(
            ErrorKind::InvalidInput,
            "Cloud Auth is not enabled for Development Identities".to_string(),
        ))
    }
    fn sync_url(&self) -> Result<String, DittoError> {
        Err(DittoError::new(
            ErrorKind::InvalidInput,
            "Cloud sync is not enabled for Development Identities".to_string(),
        ))
    }

    fn authenticator(&self) -> Option<DittoAuthenticator> {
        Some(self.auth.clone())
    }

    fn make_listener(&self, transports: Arc<RwLock<Transports>>) -> Option<Arc<ValidityListener>> {
        Some(ValidityListener::new(transports, self.auth.auth_client()))
    }
}

impl SharedKey {
    /// Construct a `SharedKey` Identity
    /// * `ditto_root` - An instance of `DittoRoot` for local storage
    /// * `app_id` - The unique UUIDv4 identifying this app; must be shared
    ///   across all instances
    /// * `key_der_b64` - A pre-shared key for securing peer-to-peer
    ///   communication encoded as a Base64 string
    pub fn new(
        ditto_root: Arc<dyn DittoRoot>,
        app_id: AppId,
        key_der_b64: &str,
    ) -> Result<Self, DittoError> {
        let c_data_dir = ditto_root.data_dir_to_c_str()?;
        let c_app_id = app_id.to_c_string();
        let c_key_der = char_p::new(key_der_b64);
        let auth_client = unsafe {
            ffi_sdk::ditto_auth_client_make_with_shared_key(
                Some(c_data_dir.as_ref()),
                c_app_id.as_ref(),
                c_key_der.as_ref(),
                0,
            )
            .ok_or(ErrorKind::Config)?
        };
        let authenticator = DittoAuthenticator::new(Arc::new(auth_client));
        Ok(Self {
            ditto_root,
            app_id,
            auth: authenticator,
        })
    }
}

impl Identity for SharedKey {
    fn site_id(&self) -> SiteId {
        unsafe { ffi_sdk::ditto_auth_client_get_site_id(&self.auth_client()) }
    }

    fn auth_client(&self) -> Arc<BoxedAuthClient> {
        self.auth.auth_client()
    }

    fn app_id(&self) -> Option<&AppId> {
        Some(&self.app_id)
    }

    fn ditto_root(&self) -> Arc<dyn DittoRoot> {
        self.ditto_root.clone()
    }

    fn is_web_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_web_valid(&self.auth_client()) != 0 }
    }

    fn is_x509_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_x509_valid(&self.auth_client()) != 0 }
    }

    fn auth_url(&self) -> Result<String, DittoError> {
        let msg = "SharedKey Identities don't support auth urls".to_owned();
        Err(DittoError::new(ErrorKind::Config, msg))
    }

    fn sync_url(&self) -> Result<String, DittoError> {
        let msg = "SharedKey Identities to support cloud sync".to_owned();
        Err(DittoError::new(ErrorKind::Config, msg))
    }

    fn authenticator(&self) -> Option<DittoAuthenticator> {
        None
    }

    fn make_listener(&self, _transports: Arc<RwLock<Transports>>) -> Option<Arc<ValidityListener>> {
        None
    }

    fn is_cloud_sync_enabled(&self) -> bool {
        false
    }
}

impl Manual {
    /// Contruct a Manual Identity
    /// * `ditto_root` - DittoRoot instance indicating local storage directory
    /// * `certificate_config_b64` - A valid configuration of a x509 PKI Client
    ///   Certificate Chain, Private Key, and other Identity data which
    ///   identifies this instance of this app to other peers and allows for a
    ///   TLS session to be established. This bundle should be provided as a
    ///   single Base64 encoded string
    pub fn new(
        ditto_root: Arc<dyn DittoRoot>,
        certificate_config_b64: &str,
    ) -> Result<Self, DittoError> {
        let certificate_config = char_p::new(certificate_config_b64);
        let auth_client = unsafe {
            ffi_sdk::ditto_auth_client_make_with_static_x509(certificate_config.as_ref())
                .ok_or(ErrorKind::Config)?
        };
        let authenticator = DittoAuthenticator::new(Arc::new(auth_client));
        Ok(Self {
            ditto_root,
            auth: authenticator,
        })
    }
}

impl Identity for Manual {
    fn site_id(&self) -> SiteId {
        unsafe { ffi_sdk::ditto_auth_client_get_site_id(&self.auth_client()) }
    }

    fn auth_client(&self) -> Arc<BoxedAuthClient> {
        self.auth.auth_client()
    }

    fn app_id(&self) -> Option<&AppId> {
        None
    }

    fn ditto_root(&self) -> Arc<dyn DittoRoot> {
        self.ditto_root.clone()
    }

    fn is_web_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_web_valid(&self.auth_client()) != 0 }
    }

    fn is_x509_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_x509_valid(&self.auth_client()) != 0 }
    }

    fn auth_url(&self) -> Result<String, DittoError> {
        let msg = "Manual Identities don't support auth urls".to_owned();
        Err(DittoError::new(ErrorKind::Config, msg))
    }

    fn sync_url(&self) -> Result<String, DittoError> {
        let msg = "Manual Identities to support cloud sync".to_owned();
        Err(DittoError::new(ErrorKind::Config, msg))
    }

    fn authenticator(&self) -> Option<DittoAuthenticator> {
        None
    }

    fn make_listener(&self, _transports: Arc<RwLock<Transports>>) -> Option<Arc<ValidityListener>> {
        None
    }

    fn is_cloud_sync_enabled(&self) -> bool {
        false
    }
}

impl OnlinePlayground {
    /// Construct a new OnlinePlayground identity.
    /// * `ditto_root` - Instance of DittoRoot indicating local storage
    ///   directory
    /// * `app_id` - A unique AppId which must be a valid UUIDv4
    /// * `enable_cloud_sync` - Should WebSocket sync with
    ///   wss://<app_id>.cloud.ditto.live be
    /// enabled by default. Do not enable this if you want to provide a custom
    /// sync URL later
    /// * `custom_auth_url` - An optional Alternative URL for authentication
    ///   requests. Defaults to
    /// `https://<app_id>.cloud.ditto.live/`
    pub fn new(
        ditto_root: Arc<dyn DittoRoot>,
        app_id: AppId,
        enable_cloud_sync: bool,
        custom_auth_url: Option<&str>,
    ) -> Result<Self, DittoError> {
        let c_data_dir = ditto_root.data_dir_to_c_str()?;
        let c_app_id = app_id.to_c_string();
        let auth_url = match custom_auth_url {
            Some(url) => url.to_owned(),
            None => app_id.default_auth_url(),
        };
        // This will change to ditto_auth_client_make_with_web shortly
        // The loginprovider, etc. will be constructed here
        // and communicate with the 'anonymous' AuthProvider on the Cloud side
        let auth_client = unsafe {
            ffi_sdk::ditto_auth_client_make_for_development(
                Some(c_data_dir.as_ref()),
                c_app_id.as_ref(),
                0,
            )
            .ok_or(ErrorKind::Config)?
        };
        let authenticator = DittoAuthenticator::new(Arc::new(auth_client));
        Ok(Self {
            ditto_root,
            app_id,
            auth: authenticator,
            enable_cloud_sync,
            auth_url,
        })
    }
}

impl Identity for OnlinePlayground {
    fn site_id(&self) -> SiteId {
        unsafe { ffi_sdk::ditto_auth_client_get_site_id(&self.auth_client()) }
    }

    fn auth_client(&self) -> Arc<BoxedAuthClient> {
        self.auth.auth_client()
    }

    fn app_id(&self) -> Option<&AppId> {
        Some(&self.app_id)
    }

    fn ditto_root(&self) -> Arc<dyn DittoRoot> {
        self.ditto_root.clone()
    }

    fn is_web_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_web_valid(&self.auth_client()) != 0 }
    }

    fn is_x509_valid(&self) -> bool {
        unsafe { ffi_sdk::ditto_auth_client_is_x509_valid(&self.auth_client()) != 0 }
    }

    fn auth_url(&self) -> Result<String, DittoError> {
        Ok(self.auth_url.to_owned())
    }

    fn sync_url(&self) -> Result<String, DittoError> {
        Ok(self.app_id.default_sync_url())
    }

    fn authenticator(&self) -> Option<DittoAuthenticator> {
        Some(self.auth.clone())
    }

    fn make_listener(&self, transports: Arc<RwLock<Transports>>) -> Option<Arc<ValidityListener>> {
        Some(ValidityListener::new(transports, self.auth.auth_client()))
    }

    fn is_cloud_sync_enabled(&self) -> bool {
        self.enable_cloud_sync
    }
}

mod private {
    use super::*;
    pub trait Sealed {}
    impl Sealed for OnlinePlayground {}
    impl Sealed for Manual {}
    impl Sealed for Online {}
    impl Sealed for Development {}
    impl Sealed for SharedKey {}
}
