//! Time Series and Event capabilities

use_prelude!();

use chrono::{DateTime, Utc};

use crate::error::{DittoError, ErrorKind};

pub struct TimeSeries {
    pub(super) ditto: Arc<ffi_sdk::BoxedDitto>,
    pub(super) ts_name: char_p::Box,
}

impl TimeSeries {
    pub fn insert<T: ::serde::Serialize>(
        &self,
        datetime: &DateTime<Utc>,
        content: T,
    ) -> Result<(), DittoError> {
        let cbor = ::serde_cbor::to_vec(content.borrow())?;
        let ditto_time = DittoTimestamp::from(datetime);
        let ts_event = unsafe {
            ffi_sdk::ditto_timeseries_event_from_cbor(
                ditto_time.epoch_bytes(),
                ditto_time.nanoseconds(),
                self.ts_name.as_ref(),
                cbor.as_slice().into(),
            )
            .ok_or(ErrorKind::InvalidInput)?
        };

        let mut txn = unsafe { ffi_sdk::ditto_write_transaction(&*self.ditto).ok()? };
        let status =
            unsafe { ffi_sdk::ditto_insert_timeseries_event(&*self.ditto, &mut txn, ts_event) };
        if status != 0 {
            return Err(DittoError::from_ffi(ErrorKind::InvalidInput));
        }
        let status = unsafe { ffi_sdk::ditto_write_transaction_commit(&*self.ditto, txn) };
        if status != 0 {
            return Err(DittoError::from_ffi(ErrorKind::Internal));
        }
        Ok(())
    }
}

#[derive(Debug)]
pub struct DittoTimestamp {
    pub epoch_sec: u64,
    pub nanos: u32,
}

impl DittoTimestamp {
    /// Construct a Timestamp key for an Event in a Timeseries from
    /// epoch seconds and nanosecond offset
    pub fn new(epoch_sec: u64, nanos: u32) -> Self {
        DittoTimestamp { epoch_sec, nanos }
    }

    /// Utility to make a timestamp from the current UTC time
    pub fn utc_now() -> Self {
        let datetime = Utc::now();
        let epoch_sec: u64 = datetime.timestamp() as u64;
        let nanos: u32 = datetime.timestamp_subsec_nanos();
        Self::new(epoch_sec, nanos)
    }

    pub fn epoch_bytes(&self) -> [u8; 8] {
        self.epoch_sec.to_be_bytes()
    }

    pub fn nanoseconds(&self) -> u32 {
        self.nanos
    }
}

impl From<&::chrono::DateTime<Utc>> for DittoTimestamp {
    fn from(datetime: &::chrono::DateTime<Utc>) -> DittoTimestamp {
        let epoch_sec: u64 = datetime.timestamp() as u64;
        let nanos: u32 = datetime.timestamp_subsec_nanos();
        DittoTimestamp::new(epoch_sec, nanos)
    }
}
