use std::{collections::HashMap, path::PathBuf, sync::Arc};

use ffi_sdk::{BoxedAttachmentHandle, BoxedDitto};
use serde::ser::SerializeMap;

use super::ditto_attachment_token::DittoAttachmentToken;

#[derive(Debug)]
pub struct DittoAttachment {
    id: Box<[u8]>,
    len: u64,
    metadata: HashMap<String, String>,
    ditto: Arc<BoxedDitto>,
    attachment_handle: BoxedAttachmentHandle,
}

impl serde::Serialize for DittoAttachment {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let mut map = serializer.serialize_map(Some(4))?;
        map.serialize_entry("_type", "ditto_attachment")?;
        map.serialize_entry("_id", &self.id)?;
        map.serialize_entry("_len", &self.len)?;
        map.serialize_entry("_meta", &self.metadata)?;
        map.end()
    }
}

impl DittoAttachment {
    pub fn new(
        id: Box<[u8]>,
        len: u64,
        metadata: HashMap<String, String>,
        ditto: Arc<BoxedDitto>,
        attachment_handle: BoxedAttachmentHandle,
    ) -> Self {
        Self {
            id,
            len,
            metadata,
            ditto,
            attachment_handle,
        }
    }

    pub fn new_with_token(
        token: DittoAttachmentToken,
        ditto: Arc<BoxedDitto>,
        attachment_handle: BoxedAttachmentHandle,
    ) -> Self {
        Self {
            id: token.id,
            len: token.len,
            metadata: token.metadata,
            ditto,
            attachment_handle,
        }
    }

    pub fn path(&self) -> PathBuf {
        let p = unsafe {
            ffi_sdk::ditto_get_complete_attachment_path(&self.ditto, &self.attachment_handle)
        };
        let p_string = p.to_string();
        p_string.into()
    }
}

#[cfg(test)]
mod tests {
    use std::collections::HashMap;

    use serde_json::json;

    use crate::{
        error::DittoError,
        prelude::*,
        store::{
            ditto_attachment_fetch_event::DittoAttachmentFetchEvent,
            ditto_attachment_token::DittoAttachmentToken,
        },
    };

    fn setup_ditto() -> Result<Ditto, DittoError> {
        let mut ditto = Ditto::builder()
            .with_temp_dir()
            .with_identity(|ditto_root| identity::Development::random(ditto_root))?
            .with_minimum_log_level(CLogLevel::Info)
            .build()?;
        ditto.set_license_from_env("DITTO_LICENSE")?;
        Ok(ditto)
    }

    #[test]
    fn attachment_serialize() {
        let ditto = setup_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();

        let original_test_file_path = "tests/data/attachment_file_1.txt";

        let metadata = {
            let mut m = HashMap::new();
            m.insert("key_1".to_string(), "value_1".to_string());
            m.insert("key_2".to_string(), "value_2".to_string());
            m
        };

        let attachment = collection
            .new_attachment(original_test_file_path, metadata.clone())
            .expect("new_attachment");
        let attachment_id = attachment.id.clone();
        let attachment_len = attachment.len;
        let attachment_file_path = attachment.path();
        assert_ne!(
            original_test_file_path,
            attachment_file_path
                .clone()
                .into_os_string()
                .into_string()
                .unwrap()
        );

        let content = json!({"hello": "again"});
        let collection = store.collection("test").unwrap();
        let id = collection.insert(content, None, false).unwrap();
        let mut doc = collection.find_by_id(id).exec().unwrap();

        let set = doc.set("att", attachment);
        assert!(set.is_ok());

        let attachment_token = doc.get::<DittoAttachmentToken>("att").unwrap();
        assert_eq!(attachment_token.id, attachment_id);
        assert_eq!(attachment_token.len, attachment_len);
        assert_eq!(attachment_token.metadata, metadata);

        let test_file = std::fs::read(original_test_file_path).unwrap();
        let attachment_file = std::fs::read(attachment_file_path).unwrap();

        assert_eq!(test_file, attachment_file);

        assert_eq!(test_file.len() as u64, attachment_len);
    }

    #[tokio::test(flavor = "multi_thread", worker_threads = 1)]
    async fn attachment_fetch() {
        let ditto = setup_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();

        let original_test_file_path = "tests/data/attachment_file_1.txt";

        let attachment = collection
            .new_attachment(original_test_file_path, HashMap::new())
            .expect("new_attachment");

        let content = json!({"hello": "again"});
        let collection = store.collection("test").unwrap();
        let id = collection.insert(content, None, false).unwrap();
        let mut doc = collection.find_by_id(id).exec().unwrap();

        let set = doc.set("att", attachment);
        assert!(set.is_ok());

        let attachment_token = doc.get::<DittoAttachmentToken>("att").unwrap();
        let (completed_sender, mut completed_receiver) = tokio::sync::mpsc::unbounded_channel();
        let _fetcher = collection
            .fetch_attachment(attachment_token, move |event| {
                if let DittoAttachmentFetchEvent::Completed { attachment } = event {
                    let _ = completed_sender.send(attachment);
                }
            })
            .unwrap();

        let fetched_attachment = completed_receiver.recv().await.unwrap();
        let attachment_file_path = fetched_attachment.path();

        let test_file = std::fs::read(original_test_file_path).unwrap();
        let attachment_file = std::fs::read(attachment_file_path).unwrap();

        assert_eq!(test_file, attachment_file);
    }
}
