//! Detect if LibDitto is present and if not, download it
//! Derived from openssl-sys build.rs
//! The search order is as follows:
//! 1) DITTOFFI_SEARCH_PATH
//! 2) OUT_DIR (target/<profile>/build/dittolive-ditto-sys-.../out)
//! 3) PWD
//! 4) CARGO_BUILD_TARGET_DIR (built into Cargo, used by CI)
//! 5) Host built-in defaults (ie. /usr/lib, /lib, /usr/local/lib, $HOME/lib)
//! controlled by linker

use ::std::{
    env,
    ffi::OsString,
    io::ErrorKind,
    path::{Path, PathBuf},
};

const LIB_NAME: &str = "dittoffi";
const SEARCH_PATH_ENV_VAR: &str = "DITTOFFI_SEARCH_PATH";

pub fn main() -> ::std::io::Result<()> {
    let out_dir = env("OUT_DIR").map(PathBuf::from).unwrap();
    let version = lib_version();
    for &env_var in &[
        "CARGO_FEATURE_DOWNLOAD", // this one may not be necessary
        SEARCH_PATH_ENV_VAR,
        "LIBDITTO_STATIC",
    ] {
        println!("cargo:rerun-if-env-changed={}", env_var);
    }
    let target = env::var("TARGET").unwrap(); // compilation target
    let lib_kind = lib_kind(); // static or dylib
    let lib_name = libname_from_target(&target, &lib_kind); // libname given target and kind
    let lib_path = lib_path(&lib_name); // where libditto SHOULD be

    // don't rerun if download flag changes
    let force_download = match env::var("CARGO_FEATURE_DOWNLOAD")
        .as_ref()
        .map(|s| s.as_str())
        .ok()
    {
        Some("0") => false,
        Some(_) => true,
        None => false,
    };

    if force_download || lib_path.starts_with(out_dir) {
        download_dittoffi(lib_path.as_path(), &version, &target, &lib_name);
    }

    let lib_dir = lib_path.parent().unwrap();

    println!("cargo:rustc-link-search=native={}", lib_dir.display());

    // println!("cargo:rustc-link-lib={}={}", lib_kind, LIB_NAME);

    if lib_kind == "static" && target.contains("linux-gnu") {
        // macOS static libraries need the Security.framework for the ring crate
        println!("cargo:rustc-link-lib=dylib=dbus-1");
    }
    if target.contains("linux-gnueabihf") {
        // don't do this for x86_64 linux
        env::set_var("LD_LIBRARY_PATH", lib_dir.as_os_str()); // makes searching
                                                              // work out of the
                                                              // box on ARM
    }
    if lib_kind == "static" && target.contains("apple") {
        // macOS static libraries need the Security.framework for the ring crate
        println!("cargo:rustc-link-lib=framework=Security");
    }
    if lib_kind == "dylib" && target.contains("apple") {
        // Same path as the Executable
        println!("cargo:rustc-link-search=dylib=@rpath/");
    }
    Ok(())
}

fn lib_path(lib_name: &str) -> PathBuf {
    if let Some(mut lib_dir) = env::var_os(SEARCH_PATH_ENV_VAR).map(PathBuf::from) {
        if lib_dir.metadata().is_ok() && lib_dir.is_dir() {
            lib_dir.push(lib_name);
            if lib_dir.exists() && lib_dir.is_file() {
                return lib_dir;
            }
        }
    }

    let paths = candidate_locations(lib_name);
    for path in paths.into_iter() {
        println!("Checking for {} at {}", LIB_NAME, path.display());
        if path.exists() && path.is_file() {
            return path;
        }
    }

    let mut out_dir = env::var_os("OUT_DIR").map(PathBuf::from).unwrap();
    out_dir.push(lib_name);
    out_dir
}

// nominate some candidate locations to search for the library
fn candidate_locations(lib_name: &str) -> Vec<PathBuf> {
    let mut locations = Vec::new();
    // First we'll check three common locations used by Cargo
    // Typically CARGO_BUILD_TARGET_DIR is not available to build.rs scripts
    if let Some(mut target_dir) = env::var_os("CARGO_BUILD_TARGET_DIR").map(PathBuf::from) {
        target_dir.push(lib_name);
        locations.push(target_dir);
    }
    // Check the current working directory, typically the directory containing the
    // final executable
    if let Ok(current_dir) = env::current_dir() {
        let mut path = PathBuf::from(current_dir);
        path.push(lib_name);
        locations.push(path);
    }

    // Check the OUT_DIR, which is where the library will be downloaded if it can't
    // be found
    if let Some(out_dir) = env::var_os("OUT_DIR").map(PathBuf::from) {
        // A hack to try and get to the TARGET_DIR which is not typically passed to
        // build scripts but is used during local development and the CI
        // pipeline
        let mut target_dir = PathBuf::from(out_dir);
        target_dir.pop();
        target_dir.pop();
        target_dir.pop(); // should be target/<profile>
        println!("TARGET_DIR may be {}", target_dir.display());
        target_dir.push(lib_name);
        locations.push(target_dir);
    }
    // Now try pkg_config if available on Linux or Mac
    if let Some(path) = try_pkg_config() {
        locations.push(path);
    }
    // Now try VCPKG if on windows
    if let Some(path) = try_vcpkg() {
        locations.push(path);
    }
    // always put this one last
    let mut out_dir = env("OUT_DIR").map(PathBuf::from).unwrap(); // always present
    out_dir.push(lib_name);
    locations.push(out_dir);
    locations
}

fn env_inner(name: &str) -> Option<OsString> {
    let var = env::var_os(name);
    println!("cargo:rerun-if-env-changed={}", name);
    match var {
        Some(ref v) => println!("{} = {}", name, v.to_string_lossy()),
        None => println!("{} unset", name),
    }
    var
}

fn env(name: &str) -> Option<OsString> {
    let prefix = env::var("TARGET").unwrap().to_uppercase().replace("-", "_");
    let prefixed = format!("{}_{}", prefix, name);
    env_inner(&prefixed).or_else(|| env_inner(&name))
}

fn try_pkg_config() -> Option<PathBuf> {
    let target = env::var("TARGET").unwrap();
    let host = env::var("HOST").unwrap();
    // set cross-compile options here
    if target.contains("arm") && host.contains("x86_64") {
        env::set_var("PKG_CONFIG_ALLOW_CROSS", "1");
    }
    if target.contains("windows-gnu") && host.contains("windows") {
        env::set_var("PKG_CONFIG_ALLOW_CROSS", "1");
    } else if target.contains("windows") {
        env::set_var("PKG_CONFIG_ALLOW_CROSS", "1");
        return None;
    }

    match pkg_config::Config::new()
        .print_system_libs(false)
        .probe(LIB_NAME)
    {
        Ok(lib) => lib.link_paths.first().map(|t| t.to_owned()),
        Err(e) => {
            println!("run pkg-config failed: {:?}", e);
            None
        }
    }
}

#[cfg(target_env = "msvc")]
fn try_vcpkg() -> Option<PathBuf> {
    let lib = vcpkg::Config::new()
        .emit_includes(false)
        .find_package(LIB_NAME);
    match lib {
        Ok(lib) => Some(lib),
        Err(e) => {
            println!("vcpkg did not find Ditto library: {}", e);
            None
        }
    }
}

#[cfg(not(target_env = "msvc"))]
fn try_vcpkg() -> Option<PathBuf> {
    None
}

// the REQUESTED kind of library
// If the library of one type isn't present we'll switch to the other
fn lib_kind() -> &'static str {
    match env::var("LIBDITTO_STATIC")
        .as_ref()
        .map(|s| s.as_str())
        .ok()
    {
        Some("0") => "dylib",
        Some(_) => "static",
        None => "static",
    }
}

// To suppot cross-compilation only the specified TARGET should be used
// to determine the version of libdittoffi to fetch
fn libname_from_target(target: &str, kind: &str) -> &'static str {
    let mut libname = "libdittoffi.a";
    if target.contains("apple") && kind == "dylib" {
        libname = "libdittoffi.dylib";
    } else if target.contains("linux") && kind == "dylib" {
        libname = "libdittoffi.so";
    } else if target.contains("windows") && kind == "dylib" {
        libname = "dittoffi.dll";
    } else if target.contains("windows") && kind == "static" {
        libname = "dittoffi.lib";
    }
    libname
}

fn lib_version() -> String {
    env::var("CARGO_PKG_VERSION").unwrap()
}

fn download_dittoffi(output_path: &Path, version: &str, target: &str, lib_name: &str) {
    // Always use a release build of libdittoffi
    let url = format!(
        "https://software.ditto.live/rust/Ditto/{version}/{target}/release/{libname}",
        target = target,
        version = version,
        libname = lib_name,
    );

    match output_path.metadata() {
        Ok(_) => {
            println!(
                "{} is already downloaded at {}",
                &lib_name,
                output_path.display()
            );
        }
        Err(e) => {
            match e.kind() {
                ErrorKind::NotFound => {
                    println!("Attempting to download {}", &url);
                    let _output = std::process::Command::new("curl")
                        .arg("-o")
                        .arg(output_path)
                        .arg("--create-dirs")
                        .arg("--location")
                        .arg("--compressed")
                        .arg("--max-time")
                        .arg("1024")
                        .arg(url)
                        .output() // wait for output
                        .expect(
                            "Failed to download Ditto SDK binary component for compilation target",
                        );
                }
                _ => {
                    println!(
                        "Unexpected error when downloading {} to {}",
                        &lib_name,
                        &output_path.display()
                    );
                }
            }
        }
    }
}
