//! Detect if LibDitto is present and if not, download it
//! Derived from openssl-sys build.rs
//! The search order is as follows:
//! 1) DITTOFFI_SEARCH_PATH
//! 2) CARGO_MANIFEST_DIR
//! 3) PWD
//! 4) Common Posix Paths
//! 5) pkg_config
//! 6) vcpkg
//! 7) Download

use ::std::{
    collections::HashSet,
    env,
    ffi::OsString,
    io::ErrorKind,
    path::{Path, PathBuf},
};

const LIB_NAME: &str = "dittoffi";
const DEFAULT_KIND: &str = "dylib";
const SEARCH_PATH_ENV_VAR: &str = "DITTOFFI_SEARCH_PATH";

pub fn main() -> ::std::io::Result<()> {
    for &env_var in &[
        "CARGO_FEATURE_DOWNLOAD", // this one may not be necessary
        SEARCH_PATH_ENV_VAR,
        "LIBDITTO_STATIC",
    ] {
        println!("cargo:rerun-if-env-changed={}", env_var);
    }
    let target = env::var("TARGET").unwrap();

    // don't rerun if download flag changes
    let download = match env::var("CARGO_FEATURE_DOWNLOAD")
        .as_ref()
        .map(|s| s.as_str())
        .ok()
    {
        Some("0") => false,
        Some(_) => true,
        None => false,
    };

    if download {
        // force download of libditto
        download_dittoffi(&target);
    }

    let lib_dir = find_libditto(&target); // where libditto SHOULD be

    if !lib_dir.is_dir() {
        println!(
            "Ditto Library Directory {} is not a directory! This is incorrect.",
            &lib_dir.display()
        );
    }
    println!(
        "cargo:rustc-link-search=native={}",
        lib_dir.to_string_lossy()
    );
    // TODO - parse the version of libditto and check compatible
    let libs_env = env("DITTO_LIBS");
    let libs = match libs_env.as_ref().and_then(|s| s.to_str()) {
        Some(ref v) => {
            if v.is_empty() {
                vec![]
            } else {
                v.split(':').collect()
            }
        }
        None => vec![LIB_NAME],
    };

    let kind = determine_mode(Path::new(&lib_dir), &libs);
    for lib in libs.into_iter() {
        println!("cargo:rustc-link-lib={}={}", kind, lib);
    }
    // if kind == "static" && target.contains("windows") { Do Windows specific
    // things}
    Ok(())
}

// Returns the directory containing the library
fn find_libditto(target: &str) -> PathBuf {
    let lib_dir = env::var_os(SEARCH_PATH_ENV_VAR).map(PathBuf::from);
    match lib_dir {
        Some(lib_dir) => lib_dir,
        // this is our key difference
        // if the search dir was not explicitly defined,
        // we will search common places
        // otherwise we default to the OUT_DIR which is always present
        None => match find_libditto_dir(target) {
            Some(p) => {
                assert!(p.is_dir());
                p
            }
            None => {
                // we use cargo manifest dir because OUT_DIR changes regularly
                let out_dir = env("CARGO_MANIFEST_DIR").map(PathBuf::from).unwrap();
                download_dittoffi(target);
                out_dir
            }
        },
    }
}

// returns the DIRECTORY containing libditto
fn find_libditto_dir(_target: &str) -> Option<PathBuf> {
    // First we'll check three common locations used by Cargo
    if let Ok(target_dir) = env::var("CARGO_BUILD_TARGET_DIR") {
        if let Some(dir) = resolve_with_wellknown_location(target_dir.as_str()) {
            assert!(dir.is_dir());
            return dir.into();
        }
    }

    // Check the Manifest Directory
    if let Ok(manifest_dir) = env::var("CARGO_MANIFEST_DIR") {
        if let Some(dir) = resolve_with_wellknown_location(manifest_dir.as_str()) {
            assert!(dir.is_dir());
            return dir.into();
        }
    }

    // Try a few well known locations on POSIX hosts
    // First the FHS recommended path for a shared lib
    if let Some(dir) = resolve_with_wellknown_location("/usr/local/lib") {
        return dir.into();
    } else if let Some(dir) = resolve_with_wellknown_location("/opt/pkg") {
        return dir.into();
    } else if let Some(dir) = resolve_with_wellknown_location("/opt/local") {
        return dir.into();
    // A path popular on Raspbian Linux
    } else if let Some(dir) = resolve_with_wellknown_location("/usr/lib") {
        return dir.into();
    } else if let Some(dir) = resolve_with_wellknown_location("/lib") {
        return dir.into();
    }

    // Now try pkg_config if available on Linux or Mac
    try_pkg_config();
    // Now try VCPKG if on windows
    try_vcpkg();
    None
}

fn env_inner(name: &str) -> Option<OsString> {
    let var = env::var_os(name);
    println!("cargo:rerun-if-env-changed={}", name);
    match var {
        Some(ref v) => println!("{} = {}", name, v.to_string_lossy()),
        None => println!("{} unset", name),
    }
    var
}

fn env(name: &str) -> Option<OsString> {
    let prefix = env::var("TARGET").unwrap().to_uppercase().replace("-", "_");
    let prefixed = format!("{}_{}", prefix, name);
    env_inner(&prefixed).or_else(|| env_inner(&name))
}

fn determine_mode(lib_dir: &Path, libs: &[&str]) -> &'static str {
    // guard for malformed paths
    if let Err(_e) = lib_dir.metadata() {
        return "dylib";
    }
    println!(
        "Confirming Ditto library present in {}",
        &lib_dir.to_string_lossy()
    );
    let kind = env("LIBDITTO_STATIC");
    match kind.as_ref().and_then(|s| s.to_str()).map(|s| &s[..]) {
        Some("0") => return "dylib",
        Some(_) => return "static",
        None => {}
    }

    let files = lib_dir
        .read_dir()
        .unwrap()
        .map(|e| e.unwrap())
        .map(|e| e.file_name())
        .filter_map(|e| e.into_string().ok())
        .collect::<HashSet<_>>();
    let can_static = libs
        .iter()
        .all(|l| files.contains(&format!("lib{}.a", l)) || files.contains(&format!("{}.lib", l)));
    let can_dylib = libs.iter().all(|l| {
        files.contains(&format!("lib{}.so", l))
            || files.contains(&format!("{}.dll", l))
            || files.contains(&format!("lib{}.dylib", l))
    });
    match (can_static, can_dylib) {
        (true, false) => return "static",
        (false, true) => return "dylib",
        (false, false) => {
            // panic!(
            //     "Ditto library search directory at `{}` does not contain the
            // required files to \      either statically or
            // dynamically link to the Ditto SDK",     lib_dir.
            // display() );
        }
        (true, true) => {}
    }
    // If no preference is specified
    // return a default
    DEFAULT_KIND
}

// Checks the provided directory actually contains libdittoffi
fn resolve_with_wellknown_location(dir: &str) -> Option<PathBuf> {
    let root_dir = Path::new(dir);
    if let Ok(files) = root_dir.read_dir() {
        files
            .map(|e| e.unwrap())
            .map(|e| e.file_name())
            .filter_map(|e| e.into_string().ok())
            .find(|x| x.contains(LIB_NAME))
            .map(|_x| {
                let path = PathBuf::from(root_dir); // full path of lib ditto
                assert!(path.is_absolute());
                path
            })
    } else {
        None
    }
}

fn try_pkg_config() {
    let target = env::var("TARGET").unwrap();
    let host = env::var("HOST").unwrap();
    if target.contains("windows-gnu") && host.contains("windows") {
        env::set_var("PKG_CONFIG_ALLOW_CROSS", "1");
    } else if target.contains("windows") {
        return;
    }

    let _lib = match pkg_config::Config::new()
        .print_system_libs(false)
        .probe(LIB_NAME)
    {
        Ok(lib) => lib,
        Err(e) => {
            println!("run pkg-config failed: {:?}", e);
            return;
        }
    };
    // don't panic as we can try to download still
}

#[cfg(target_env = "msvc")]
fn try_vcpkg() {
    let lib = vcpkg::Config::new()
        .emit_includes(false)
        .find_package(LIB_NAME);
    if let Err(e) = lib {
        println!("Note: vcpkg did not find Ditto library: {}", e);
        return;
    };
    let lib = lib.unwrap();
    // Include any other system libraries required here
    // Don't panic as we can try to download still
}

#[cfg(not(target_env = "msvc"))]
fn try_vcpkg() {}

// To suppot cross-compilation only the specified TARGET should be used
// to determine the version of libdittoffi to fetch
fn libname_from_target() -> &'static str {
    let target = env::var("TARGET").unwrap();
    let kind = match env::var("LIBDITTO_STATIC")
        .as_ref()
        .map(|s| s.as_str())
        .ok()
    {
        Some("0") => "dylib",
        Some(_) => "static",
        None => "dylib", // default to dylib as its generally better supported
    };

    let mut libname = "libdittoffi.a";
    if target.contains("apple-darwin") && kind == "dylib" {
        libname = "libdittoffi.dylib";
    } else if target.contains("linux") && kind == "dylib" {
        libname = "libdittoffi.so";
    } else if target.contains("windows") && kind == "dylib" {
        libname = "dittoffi.dll";
    } else if target.contains("windows") && kind == "static" {
        libname = "dittoffi.lib";
    }

    libname
}

fn download_dittoffi(target: &str) {
    let version = env::var("CARGO_PKG_VERSION").unwrap();
    let libname = libname_from_target();
    let lib_dir = env::var_os("OUT_DIR").unwrap();

    // Always use a release build of libdittoffi
    let url = format!(
        "https://software.ditto.live/rust/Ditto/{version}/{target}/release/{libname}",
        target = target,
        version = version,
        libname = libname,
    );

    let mut output_path = PathBuf::from(lib_dir);
    output_path.push(&libname);
    match Path::new(&output_path).metadata() {
        Ok(_) => {
            println!(
                "{} is already downloaded at {}",
                &libname,
                output_path.display()
            );
        }
        Err(e) => {
            match e.kind() {
                ErrorKind::NotFound => {
                    println!("Attempting to download {}", &url);
                    let _status = std::process::Command::new("curl")
                        .arg(url)
                        .arg("-o")
                        .arg(output_path)
                        .status() // await result
                        .expect(
                            "Failed to download Ditto SDK binary component for compilation target",
                        );
                }
                _ => {
                    println!(
                        "Unexpected error when downloading {} to {}",
                        &libname,
                        &output_path.display()
                    );
                }
            }
        }
    }
}
