//! Detect if LibDitto is present and if not, download it
//! Derived from openssl-sys build.rs
//! The search order is as follows:
//! 1) DITTOFFI_SEARCH_PATH
//! 2) OUT_DIR (target/<profile>/build/dittolive-ditto-sys-.../out)
//! 3) PWD
//! 4) CARGO_BUILD_TARGET_DIR (built into Cargo, used by CI)
//! 5) Host built-in defaults (ie. /usr/lib, /lib, /usr/local/lib, $HOME/lib)
//! controlled by linker
//!
//! It is strongly recommend internal builds use the DITTOFFI_SEARCH_PATH to
//! ensure a locally built (not downloaded) libdittoffi is used.

use ::std::{
    env,
    ffi::OsString,
    io::ErrorKind,
    path::{Path, PathBuf},
};

const LIB_NAME: &str = "dittoffi";
const SEARCH_PATH_ENV_VAR: &str = "DITTOFFI_SEARCH_PATH";

pub fn main() -> ::std::io::Result<()> {
    let out_dir = env("OUT_DIR").map(PathBuf::from).unwrap();
    let version = lib_version();
    for &env_var in &[
        "CARGO_FEATURE_DOWNLOAD", // this one may not be necessary
        SEARCH_PATH_ENV_VAR,
        "LIBDITTO_STATIC",
    ] {
        println!("cargo:rerun-if-env-changed={}", env_var);
    }
    let target = env::var("TARGET").unwrap(); // compilation target
    let lib_kind = lib_kind(); // static or dylib
    let lib_name = libname_from_target(&target, lib_kind); // libname given target and kind
    let lib_path = lib_path(lib_name); // where libditto SHOULD be

    // don't rerun if download flag changes
    let force_download = match env::var("CARGO_FEATURE_DOWNLOAD")
        .as_ref()
        .map(|s| s.as_str())
        .ok()
    {
        Some("0") => false,
        Some(_) => true,
        None => false,
    };

    if force_download || lib_path.starts_with(out_dir) {
        download_dittoffi(lib_path.as_path(), &version, &target, lib_name);
    }

    let lib_dir = lib_path.parent().unwrap();
    println!("cargo:rustc-link-lib={}={}", lib_kind, LIB_NAME);
    // linking currently forced to be static to avoid issues with dylib loader
    println!("cargo:rustc-link-search=native={}", lib_dir.display());

    if target.contains("linux-gnueabihf") {
        // don't do this for x86_64 linux
        env::set_var("LD_LIBRARY_PATH", lib_dir.as_os_str()); // makes searching
                                                              // work out of the
                                                              // box on ARM
    }
    if lib_kind == "static" && target.contains("apple") {
        // macOS static libraries need the Security.framework for the ring crate
        println!("cargo:rustc-link-lib=framework=Security");
    }
    Ok(())
}

fn lib_path(lib_name: &str) -> PathBuf {
    if let Some(mut lib_dir) = env::var_os(SEARCH_PATH_ENV_VAR).map(PathBuf::from) {
        if lib_dir.metadata().is_ok() && lib_dir.is_dir() {
            lib_dir.push(lib_name);
            if lib_dir.exists() && lib_dir.is_file() {
                return lib_dir;
            } else {
                eprintln!(
                    "Provided binary path {} does not (yet) exist!",
                    lib_dir.display(),
                );
                return lib_dir;
            }
        }
    }

    let paths = candidate_locations(lib_name);
    for path in paths.into_iter() {
        eprintln!("Checking for {} at {}", LIB_NAME, path.display());
        if path.exists() && path.is_file() {
            return path;
        }
    }
    // default to the .../deps/dittolive-ditto-sys/build/out
    // but this may not be correct for internal builds using libdittoffi from target
    // dir
    env::var_os("OUT_DIR").unwrap().with(lib_name)
}

// nominate some candidate locations to search for the library
fn candidate_locations(lib_name: &str) -> Vec<PathBuf> {
    let mut locations = Vec::new();
    // First we'll check three common locations used by Cargo
    // Typically CARGO_BUILD_TARGET_DIR is not available to build.rs scripts
    if let Some(target_dir) = env::var_os("CARGO_BUILD_TARGET_DIR").map(PathBuf::from) {
        locations.push(target_dir.with(lib_name));
    }
    // Check the current working directory, typically the directory containing the
    // final executable
    if let Ok(path) = env::current_dir() {
        locations.push(path.with(lib_name))
    }

    // Check the OUT_DIR, which is where the library will be downloaded if it can't
    // be found
    if let Some(out_dir) = env::var_os("OUT_DIR") {
        // A hack to try and get to the TARGET_DIR which is not typically passed to
        // build scripts but is used during local development and the CI
        // pipeline
        // Typical value here is
        // `./target/<profile>/build/dittolive-ditto-sys-<hash>/out`
        let mut target_dir = PathBuf::from(out_dir);
        target_dir.pop(); // out
        target_dir.pop(); // dittolive-ditto-sys-<hash>
        target_dir.pop(); // build

        // should be target/<profile>
        eprintln!("TARGET_DIR may be {}", target_dir.display());
        // The target/<profile>/deps directory often has libdittoffi as well
        // for certain internal builds
        let mut deps_dir = target_dir.clone();
        locations.push(target_dir.with(lib_name));
        deps_dir.push("deps");
        locations.push(deps_dir.with(lib_name));
    }
    // always put this one last
    let out_dir = env("OUT_DIR").map(PathBuf::from).unwrap(); // always present
    locations.push(out_dir.with(lib_name));
    locations
}

fn env_inner(name: &str) -> Option<OsString> {
    let var = env::var_os(name);
    println!("cargo:rerun-if-env-changed={}", name);
    match var {
        Some(ref v) => println!("{} = {}", name, v.to_string_lossy()),
        None => println!("{} unset", name),
    }
    var
}

fn env(name: &str) -> Option<OsString> {
    let prefix = env::var("TARGET").unwrap().to_uppercase().replace("-", "_");
    let prefixed = format!("{}_{}", prefix, name);
    env_inner(&prefixed).or_else(|| env_inner(name))
}

// the REQUESTED kind of library
// If the library of one type isn't present we'll switch to the other
fn lib_kind() -> &'static str {
    match env::var("LIBDITTO_STATIC")
        .as_ref()
        .map(|s| s.as_str())
        .ok()
    {
        Some("0") => "dylib",
        Some(_) => "static",
        None => "static",
    }
}

// To suppot cross-compilation only the specified TARGET should be used
// to determine the version of libdittoffi to fetch
fn libname_from_target(target: &str, kind: &str) -> &'static str {
    let mut libname = "libdittoffi.a";
    if target.contains("apple") && kind == "dylib" {
        libname = "libdittoffi.dylib";
    } else if target.contains("linux") && kind == "dylib" {
        libname = "libdittoffi.so";
    } else if target.contains("windows") && kind == "dylib" {
        libname = "dittoffi.dll";
    } else if target.contains("windows") && kind == "static" {
        libname = "dittoffi.lib";
    }
    libname
}

fn lib_version() -> String {
    env::var("CARGO_PKG_VERSION").unwrap()
}

fn download_dittoffi(output_path: &Path, version: &str, target: &str, lib_name: &str) {
    // Always use a release build of libdittoffi
    let url = format!(
        "https://software.ditto.live/rust/Ditto/{version}/{target}/release/{libname}",
        target = target,
        version = version,
        libname = lib_name,
    );

    match output_path.metadata() {
        Ok(_) => {
            eprintln!(
                "{} is already downloaded at {}",
                &lib_name,
                output_path.display()
            );
        }
        Err(e) => {
            match e.kind() {
                ErrorKind::NotFound => {
                    eprintln!("Attempting to download {}", &url);
                    let status = std::process::Command::new("curl")
                        .arg("-o")
                        .arg(output_path)
                        .arg("--create-dirs")
                        .arg("--location")
                        .arg("--compressed")
                        .arg("--max-time")
                        .arg("1024")
                        .arg("-f") // Force build script to fail on 404
                        .arg(url)
                        .status() // wait for output
                        .expect(
                            "Failed to download Ditto SDK binary component for compilation target",
                        );
                    if !status.success() {
                        eprintln!("Unable to download Ditto SDK binary component");
                    }
                }
                _ => {
                    println!(
                        "Unexpected error when downloading {} to {}",
                        &lib_name,
                        &output_path.display()
                    );
                }
            }
        }
    }
}

trait WithPath {
    fn with(self, path: impl AsRef<Path>) -> PathBuf;
}

impl<ImplIntoPathBuf> WithPath for ImplIntoPathBuf
where
    ImplIntoPathBuf: Into<PathBuf>,
{
    fn with(self: ImplIntoPathBuf, path: impl AsRef<Path>) -> PathBuf {
        let mut path_buf: PathBuf = self.into();
        path_buf.push(path);
        path_buf
    }
}
