use num_bigint_dig::BigUint;
use serde::de::{self, Deserialize, Deserializer, Visitor};
use serde::ser::{Serialize, Serializer};
use std::fmt;
use std::ops::{Deref, DerefMut};

#[derive(Clone, Debug, PartialEq, Eq, PartialOrd, Ord)]
#[repr(transparent)]
// wrapper type around Bigint to implement base64 decode/encode in serde
pub(crate) struct BigUintWrapper(pub BigUint);

impl BigUintWrapper {
    fn decode(s: &str) -> Result<BigUintWrapper, String> {
        base64::decode(s)
            .map(|n| BigUintWrapper(BigUint::from_bytes_be(&n)))
            .map_err(|e| e.to_string())
    }

    fn encode(&self) -> String {
        let code: Vec<u8> = self.0.to_bytes_be();
        base64::encode(code)
    }
}

impl Deref for BigUintWrapper {
    type Target = BigUint;

    fn deref(&self) -> &BigUint {
        &self.0
    }
}

impl DerefMut for BigUintWrapper {
    fn deref_mut(&mut self) -> &mut BigUint {
        &mut self.0
    }
}

impl Serialize for BigUintWrapper {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        serializer.serialize_str(&self.encode())
    }
}

struct VisitBigUint;

impl<'de> Visitor<'de> for VisitBigUint {
    type Value = BigUintWrapper;

    fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        formatter.write_str("an base64 encoded big_integer")
    }

    fn visit_str<E>(self, s: &str) -> Result<Self::Value, E>
    where
        E: de::Error,
    {
        BigUintWrapper::decode(s).map_err(E::custom)
    }
}

impl<'de> Deserialize<'de> for BigUintWrapper {
    fn deserialize<D>(deserializer: D) -> Result<BigUintWrapper, D::Error>
    where
        D: Deserializer<'de>,
    {
        deserializer.deserialize_str(VisitBigUint)
    }
}
