//! disp
//! ====
//! 
//! [![crates.io](https://img.shields.io/crates/v/disp.svg)](https://crates.io/crates/disp)
//! [![Documentation](https://docs.rs/disp/badge.svg)](https://docs.rs/disp)
//! [![Build Status](https://travis-ci.org/btwiuse/disp.svg?branch=master)](https://travis-ci.org/btwiuse/disp)
//! 
//! derive `Display` for types that implement `Debug`
//! 
//! Example
//! 
//! ```
//! mod hello {
//!     use std::fmt::Display;
//! 
//!     #[derive(Debug, disp::Display)]
//!     pub struct Hello {}
//! }
//! 
//! fn main() {
//!     let hello = hello::Hello {};
//!     println!("Debug: {hello:?}, world!");
//!     println!("Display: {hello}, world!");
//! }
//! ```
//! 
//! Output:
//! 
//! ```
//! Debug: Hello, world!
//! Display: Hello, world!
//! ```
use proc_macro::TokenStream;
use quote::quote;

#[proc_macro_derive(Display)]
pub fn display_derive(input: TokenStream) -> TokenStream {
    let ast = syn::parse(input).expect("TokenStream could not be parsed");
    impl_display_derive(&ast)
}

fn impl_display_derive(ast: &syn::DeriveInput) -> TokenStream {
    let name = &ast.ident;
    let gen = quote! {
        impl Display for #name {
            fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
                write!(f, "{:?}", self)
            }
        }
    };
    gen.into()
}

#[cfg(test)]
mod tests {
    #[test]
    fn it_works() {
        let result = 2 + 2;
        assert_eq!(result, 4);
    }
}
