use crate::file::RandomAccessFile;
use crate::primitives::file_stack::GetError::IndexOutOfBounds;

pub struct FileStack<TFile>
    where TFile: RandomAccessFile
{
    indexFile: TFile,
    dataFile: TFile
}

impl<TFile> FileStack<TFile>
    where TFile: RandomAccessFile
{
    pub fn get_item(&mut self, index: u64) -> Result<Vec<u8>, GetError> {
        if self.item_exists(index)? {
            let mut start_addr: [u8; 8] = [0; 8];
            self.indexFile.read_exact_at(&mut start_addr, index * 8)?;
            let start_addr = u64::from_le_bytes(start_addr);
            if index == self.len()? - 1 {
                let mut buf = vec![0; self.dataFile.size_from(start_addr)? as usize];
                self.dataFile.read_exact_at(buf.as_mut_slice(), start_addr)?;
                return Ok(buf.to_vec());
            } else {
                let mut end_addr: [u8; 8] = [0; 8];
                self.indexFile.read_exact_at(&mut end_addr, (index + 1) * 8)?;
                let end_addr = u64::from_le_bytes(end_addr);
                let len = end_addr - start_addr;

                let mut buf = vec![0; len as usize];
                self.dataFile.read_exact_at(buf.as_mut_slice(), start_addr);
                Ok(buf.to_vec())
            }
        } else {
            Err(IndexOutOfBounds)
        }
    }

    pub fn len(&mut self) -> Result<u64, std::io::Error> {
        Ok(self.indexFile.size()? / 8)
    }

    pub fn item_exists(&mut self, index: u64) -> Result<bool, std::io::Error> {
        Ok(index < self.len()?)
    }

    pub fn add_item(&mut self, data: &[u8]) -> Result<(), std::io::Error> {
        let start_ptr = self.dataFile.size()?;
        self.dataFile.write_all_at(data, start_ptr)?;
        let len = self.len()?;
        self.indexFile.write_all_at(&start_ptr.to_le_bytes(), len * 8)?;
        Ok(())
    }
}

enum GetError {
    IndexOutOfBounds,
    IoError(std::io::Error)
}

impl From<std::io::Error> for GetError {
    fn from(err: std::io::Error) -> Self {
        GetError::IoError(err)
    }
}