use anyhow::Context;
use derive_more::{Deref, DerefMut};
use log::debug;
use serde::{de::DeserializeOwned, Serialize};
use std::{
    any::type_name,
    fs::{self, File, OpenOptions},
    path::{Path, PathBuf},
};

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash, Deref, DerefMut)]
pub struct Diskable<S>
where
    S: Serialize + DeserializeOwned,
{
    #[deref]
    #[deref_mut]
    inner: S,
    location: PathBuf,
}

impl<S> Diskable<S>
where
    S: Serialize + DeserializeOwned,
{
    pub fn load_or(location: impl AsRef<Path>, default: S) -> anyhow::Result<Self> {
        let location = PathBuf::from(location.as_ref());
        if let Some(parent) = location.parent() {
            fs::create_dir_all(parent)?;
        }
        if let Ok(new) = OpenOptions::new()
            .write(true)
            .create_new(true)
            .open(&location)
        {
            debug!("Populating new file at {}", location.display());
            ron::ser::to_writer_pretty(new, &default, Default::default())?;
        }
        Self::load(location)
    }

    pub fn load(location: impl AsRef<Path>) -> anyhow::Result<Self> {
        let location = PathBuf::from(location.as_ref());
        let f = File::open(&location).context(format!("Couldn't open {}", location.display()))?;
        let inner = ron::de::from_reader(f).context(format!(
            "Couldn't serialize {} from {}",
            type_name::<S>(),
            location.display()
        ))?;
        Ok(Self { inner, location })
    }

    pub fn refresh(&mut self) -> anyhow::Result<()> {
        let new = Self::load(&self.location)?;
        *self = new;
        Ok(())
    }

    pub fn flush(&self) -> anyhow::Result<()> {
        let f = OpenOptions::new()
            .create(true)
            .write(true)
            .truncate(true)
            .open(&self.location)?;
        ron::ser::to_writer_pretty(f, &self.inner, Default::default())?;
        Ok(())
    }

    pub fn into_inner(self) -> S {
        let Diskable { inner, .. } = self;
        inner
    }
}
