use std::{
    any::type_name,
    fs::{self, File, OpenOptions},
    path::{Path, PathBuf},
    thread,
};

use anyhow::Context;
use log::{debug, error};
use serde::{de::DeserializeOwned, Serialize};

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Diskable<S>
where
    S: Serialize + DeserializeOwned,
{
    inner: S,
    location: PathBuf,
}

impl<S> Diskable<S>
where
    S: Serialize + DeserializeOwned,
{
    pub fn load_or(location: impl AsRef<Path>, default: S) -> anyhow::Result<Self> {
        let location = PathBuf::from(location.as_ref());
        if let Some(parent) = location.parent() {
            fs::create_dir_all(parent)?;
        }
        if let Ok(new) = OpenOptions::new().create_new(true).open(&location) {
            debug!("Populating new file at {}", location.display());
            serde_json::to_writer_pretty(new, &default)?;
        }
        Self::load(location)
    }
    pub fn load(location: impl AsRef<Path>) -> anyhow::Result<Self> {
        let location = PathBuf::from(location.as_ref());
        let f = File::open(&location).context(format!("Couldn't open {}", location.display()))?;
        let inner = serde_json::from_reader(f).context(format!(
            "Couldn't serialize {} from {}",
            type_name::<S>(),
            location.display()
        ))?;
        Ok(Self { inner, location })
    }
    pub fn refresh(&mut self) -> anyhow::Result<()> {
        let new = Self::load(&self.location)?;
        *self = new;
        Ok(())
    }
    pub fn flush(&self) -> anyhow::Result<()> {
        let f = OpenOptions::new()
            .create(true)
            .write(true)
            .truncate(true)
            .open(&self.location)?;
        serde_json::to_writer_pretty(f, &self.inner)?;
        Ok(())
    }
}

impl<S> Drop for Diskable<S>
where
    S: Serialize + DeserializeOwned,
{
    fn drop(&mut self) {
        if let Err(e) = self.flush() {
            match thread::panicking() {
                true => error!(
                    "Couldn't persist {} to {}: {}",
                    type_name::<S>(),
                    self.location.display(),
                    e
                ),
                false => panic!(
                    "Couldn't persist {} to {}: {}",
                    type_name::<S>(),
                    self.location.display(),
                    e
                ),
            }
        }
    }
}
