# Implementing Patchable

Here's a look at the `Patchable` trait.

```rust
pub trait Patchable<P> {
    fn apply_patch(&mut self, patch: P);
```

`P` is the delta encoding that was generated by the `Diffable.create_delta_towards` method.

Different types will have different patches that need to be applied in
different ways.

Let's talk a look at an `i128`, one of the more trivial `Patchable` types.

```rust
implement Patchable<Option<i128>> for i128 {
	fn apply_patch(&mut self, patch: Option<i128>) {
		if let Some(patch) = patch {
		    *self = patch;
		}
	}
}
```

Here's a slightly more complex example patch function.

```rust
enum CustomPatchExample {
    Replace(i128),
    Add(i16)
}

// This is not how the real implementation for i128 looks.
implement Patchable<CustomPatchExample> for i128 {
	fn apply_patch(&mut self, patch: CustomPatchExample) {
		match patch {
			CustomPatchExample::Replace(new) => {
			    *self = new;
			}
			CustomPatchExample::Add(add) => {
			    self += add as i128;
			}
		}
	}
}
```
