// This file is part of the didu source code.
//
// ©️ 2022 papojari <mailto:papojari-git.ovoid@aleeas.com> <https://matrix.to/#/@papojari:artemislena.eu> <https://papojari.codeberg.page>
//
// For the license information, please view the README.md file that was distributed with this source code.

use clap::Parser;
use unit_conversions::time;
use colorful::Colorful;
use colorful::Color;

#[derive(Parser, Debug)]
#[clap(name = "didu", version, about="Duration conversion between units")]
struct Args {
    /// Duration value
    #[clap(long, short = 'v')]
    input_value: f64,
    /// Duration unit
    #[clap(long, short = 'u', possible_values = ["s", "min", "h", "d", "w", "m", "y", "dec", "cen"], default_value = "s")]
    input_unit: String,
    /// What unit(s) to convert to. "all" is not a unit but means all units.
    #[clap(long, short = 'o', possible_values = ["s", "min", "h", "d", "w", "m", "y", "dec", "cen", "all"], default_value = "all")]
    output_unit: String,
    /// If the output should be just the result or also show the input.
    #[clap(long, short = 'r')]
    only_result: bool,
}

struct PrintOutputUnit {
    seconds: bool,
    minutes: bool,
    hours: bool,
    days: bool,
    weeks: bool,
    months: bool,
    years: bool,
    decades: bool,
    centuries: bool,
}

fn main() {
    let args = Args::parse();
    let input_value = args.input_value;
    let input_unit = args.input_unit;
    let output_unit = args.output_unit;

    let minutes = match input_unit.as_str() {
        "s" => time::seconds::to_minutes(input_value),
        "min" => time::seconds::to_minutes(time::minutes::to_seconds(input_value)),
        "h" => time::hours::to_minutes(input_value),
        "d" => time::days::to_minutes(input_value),
        "w" => time::weeks::to_minutes(input_value),
        "m" => time::months::to_minutes(input_value),
        "y" => time::years::to_minutes(input_value),
        "dec" => time::decades::to_minutes(input_value),
        "cen" => time::centuries::to_minutes(input_value),
        _ => panic!(),
    };
    let seconds = time::minutes::to_seconds(minutes);
    let hours = time::minutes::to_hours(minutes);
    let days = time::minutes::to_days(minutes);
    let weeks = time::minutes::to_weeks(minutes);
    let months = time::minutes::to_months(minutes);
    let years = time::minutes::to_years(minutes);
    let decades = time::minutes::to_decades(minutes);
    let centuries = time::minutes::to_centuries(minutes);

    // print the converted duration
    println!("{} {} {}", input_value.to_string().color(Color::Yellow).bold().underlined(), input_unit.color(Color::Green).bold(), "are:" );
    let options = match output_unit.as_str() {
        "s" => PrintOutputUnit {
            seconds: true,
            minutes: false,
            hours: false,
            days: false,
            weeks: false,
            months: false,
            years: false,
            decades: false,
            centuries: false,
        },
        "min" => PrintOutputUnit {
            seconds: false,
            minutes: true,
            hours: false,
            days: false,
            weeks: false,
            months: false,
            years: false,
            decades: false,
            centuries: false,
        },
        "h" => PrintOutputUnit {
            seconds: false,
            minutes: false,
            hours: true,
            days: false,
            weeks: false,
            months: false,
            years: false,
            decades: false,
            centuries: false,
        },
        "d" => PrintOutputUnit {
            seconds: false,
            minutes: false,
            hours: false,
            days: true,
            weeks: false,
            months: false,
            years: false,
            decades: false,
            centuries: false,
        },
        "w" => PrintOutputUnit {
            seconds: false,
            minutes: false,
            hours: false,
            days: false,
            weeks: true,
            months: false,
            years: false,
            decades: false,
            centuries: false,
        },
        "m" => PrintOutputUnit {
            seconds: false,
            minutes: false,
            hours: false,
            days: false,
            weeks: false,
            months: true,
            years: false,
            decades: false,
            centuries: false,
        },
        "y" => PrintOutputUnit {
            seconds: false,
            minutes: false,
            hours: false,
            days: false,
            weeks: false,
            months: false,
            years: true,
            decades: false,
            centuries: false,
        },
        "dec" => PrintOutputUnit {
            seconds: false,
            minutes: false,
            hours: false,
            days: false,
            weeks: false,
            months: false,
            years: false,
            decades: true,
            centuries: false,
        },
        "cen" => PrintOutputUnit {
            seconds: false,
            minutes: false,
            hours: false,
            days: false,
            weeks: false,
            months: false,
            years: false,
            decades: false,
            centuries: true,
        },
        "all" => PrintOutputUnit {
            seconds: true,
            minutes: true,
            hours: true,
            days: true,
            weeks: true,
            months: true,
            years: true,
            decades: true,
            centuries: true,
        },
        _ => panic!("Invalid input: {output_unit}"),
    };

    if options.seconds {
        println!("{} {}", seconds, "s".color(Color::Blue));
    }
    if options.minutes  {
        println!("{} {}", minutes, "min".color(Color::Blue));
    }
    if options.hours  {
        println!("{} {}", hours, "h".color(Color::Blue));
    }
    if options.days  {
        println!("{} {}", days, "d".color(Color::Blue));
    }
    if options.weeks  {
        println!("{} {}", weeks, "w".color(Color::Blue));
    }
    if options.months  {
        println!("{} {}", months, "m".color(Color::Blue));
    }
    if options.years  {
        println!("{} {}", years, "y".color(Color::Blue));
    }
    if options.decades  {
        println!("{} {}", decades, "dec".color(Color::Blue));
    }
    if options.centuries  {
        println!("{} {}", centuries, "cen".color(Color::Blue));
    }
}
