use crate::adapters::{BoxedDieOnce, FlatMapDie, FlattenDie, MapDie};
use crate::Fate;

/// Trait for generating a single pseudorandom value of type `T`.
pub trait DieOnce<T> {
    /// Consumes the generator and generates a pseudorandom value.
    ///
    /// The [`Fate`] is the only source of the randomness. Besides that, the generation is
    /// deterministic.
    fn roll_once(self, fate: Fate) -> T;

    /// Creates a new [`DieOnce`] by mapping the generated values of `self`.
    ///
    /// The function `f` will be applied to the generated value of `self`. The result of the
    /// function is the generated value of the new [`DieOnce`].
    fn map_once<U, F>(self, f: F) -> MapDie<T, U, Self, F>
    where
        Self: Sized,
        F: FnOnce(T) -> U,
    {
        MapDie::new(self, f)
    }

    /// Creates a new [`DieOnce`] whose value is generated by the generated [`DieOnce`] of `self`.
    fn flatten_once<U>(self) -> FlattenDie<U, T, Self>
    where
        Self: Sized,
        T: DieOnce<U>,
    {
        FlattenDie::new(self)
    }

    /// Creates a new [`DieOnce`] similar to [`map_once`], except that the mapping
    /// produces a [`DieOnce`].
    ///
    /// The function `f` will be applied to the generated value of `self`. The result of the
    /// function is a [`DieOnce`] that generate the value for the new [`DieOnce`].
    ///
    /// It is semantically equivalent to `self.map_once(f).flatten_once()`.
    ///
    /// [`map_once`]: DieOnce::map_once
    fn flat_map_once<U, DU, F>(self, f: F) -> FlatMapDie<T, U, Self, DU, F>
    where
        Self: Sized,
        DU: DieOnce<U>,
        F: FnOnce(T) -> DU,
    {
        FlatMapDie::new(self, f)
    }

    /// Puts `self` behind a [`Box`] pointer.
    fn boxed_once<'a>(self) -> BoxedDieOnce<'a, T>
    where
        Self: Sized + 'a,
    {
        BoxedDieOnce::new(self)
    }
}
