// License: see LICENSE file at root directory of `master` branch

//! # Dia IP range
//!
//! ## Project
//!
//! - Repository: <https://bitbucket.org/haibison/ip-range>
//! - License: Nice License 1.0.0 _(see LICENSE file at root directory of `master` branch)_
//! - _This project follows [Semantic Versioning 2.0.0]_
//!
//! ## Features
//!
//! This crate provides iterators for IPv4 addresses. This is useful for programs operating on local area networks (LANs).
//!
//! For examples, see [`IPv4RangeIter`][struct::IPv4RangeIter].
//!
//! ## Notes
//!
//! Documentation is built with all features. Some of them are optional. If you see some components from other crates, you can view source to
//! see what features are required.
//!
//! [Semantic Versioning 2.0.0]: https://semver.org/spec/v2.0.0.html
//!
//! [struct::IPv4RangeIter]: struct.IPv4RangeIter.html

#![warn(missing_docs)]
#![no_std]
#![deny(unsafe_code)]

// ╔═════════════════╗
// ║   IDENTIFIERS   ║
// ╚═════════════════╝

macro_rules! code_name  { () => { "dia-ip-range" }}
macro_rules! version    { () => { "0.7.0" }}

/// # Crate name
pub const NAME: &str = "dia-ip-range";

/// # Crate code name
pub const CODE_NAME: &str = code_name!();

/// # ID of this crate
pub const ID: &str = concat!(
    "9b310d4b-54f515c0-112415bd-be4db60e-2e6fd1c8-919c250f-a92462fb-90297406-",
    "06631349-7e828af7-f5928057-9c4a026c-51cc3702-02d14ce4-79cb109a-01ab7292",
);

/// # Crate version
pub const VERSION: &str = version!();

/// # Crate release date (year/month/day)
pub const RELEASE_DATE: (u16, u8, u8) = (2021, 10, 20);

/// # Tag, which can be used for logging...
pub const TAG: &str = concat!(code_name!(), "::9b310d4b::", version!());

// ╔════════════════════╗
// ║   IMPLEMENTATION   ║
// ╚════════════════════╝

extern crate alloc;

#[cfg(any(feature="iter", feature="std"))]
extern crate std;

/// # Makes new Error with formatted string, or without one
macro_rules! err {
    () => {
        crate::Error::new(line!(), module_path!(), None)
    };
    ($s: literal) => {
        crate::Error::new(line!(), module_path!(), Some(alloc::borrow::Cow::Borrowed($s)))
    };
    ($s: literal, $($arg: tt)+) => {
        crate::Error::new(line!(), module_path!(), Some(alloc::borrow::Cow::Owned(alloc::format!($s, $($arg)+))))
    };
}

#[test]
fn test_macro_err() {
    use alloc::borrow::Cow;

    macro_rules! s_test { () => { "test" }}

    fn eq(first: Error, second: Error) -> bool {
        first.line() == second.line() && first.module_path() == second.module_path() && first.msg() == second.msg()
    }

    assert!(eq(err!(), Error::new(line!(), module_path!(), None)));
    assert!(eq(err!("test"), Error::new(line!(), module_path!(), Some(Cow::Borrowed(s_test!())))));
    assert!(eq(err!("{s:?}", s=s_test!()), Error::new(line!(), module_path!(), Some(Cow::Owned(alloc::format!("{:?}", s_test!()))))));
}

mod error;
mod ip_v4_range;

pub mod version_info;

pub use self::{
    error::*,
    ip_v4_range::*,
};

/// # Result type used in this crate
pub type Result<T> = core::result::Result<T, Error>;

#[test]
fn test_crate_version() {
    assert_eq!(VERSION, env!("CARGO_PKG_VERSION"));
}
