// License: see LICENSE file at root directory of `master` branch

//! # IPv4 range iter

#![cfg(feature="iter")]

use {
    std::net::IpAddr,

    perm::{Buffer, PermIter},

    super::{Component, IPv4Range},
};

#[cfg(test)]
use {
    alloc::string::ToString,
    std::collections::HashSet,
};

/// # IPv4 range iter
///
/// **WARNING:** The order of generated items is **not** guaranteed to be constant between crate versions.
///
/// ## Examples
///
/// ```
/// use {
///     core::str::FromStr,
///     std::{
///         collections::HashSet,
///         net::IpAddr,
///     },
///
///     dia_ip_range::{IPv4Range, IPv4RangeIter},
///     perm::Buffer,
/// };
///
/// let buffer = Buffer::with_capacity(4);
/// let iter = IPv4RangeIter::new(
///     IPv4Range::from_str("10.0.0.[2,9]")?,
///     &buffer,
/// );
/// assert_eq!(
///     iter.collect::<HashSet<_>>(),
///     [
///         [10, 0, 0, 2], [10, 0, 0, 3], [10, 0, 0, 4], [10, 0, 0, 5],
///         [10, 0, 0, 6], [10, 0, 0, 7], [10, 0, 0, 8], [10, 0, 0, 9],
///     ].iter().map(|i| IpAddr::from(*i)).collect(),
/// );
///
/// # Ok::<_, dia_ip_range::Error>(())
/// ```
#[derive(Debug, Eq, PartialEq, Hash)]
pub struct IPv4RangeIter<'a> {
    iter: PermIter<'a, Component>,
}

impl<'a> IPv4RangeIter<'a> {

    /// # Makes new instance
    ///
    /// Output buffer is used for better performance.
    ///
    /// ## Panics
    ///
    /// Each buffer can only be used by one instance of `IPv4RangeIter`. Sharing it to multiple instances will cause panics.
    pub fn new(ip_v4_range: IPv4Range, buffer: &'a Buffer<u8>) -> Self {
        Self {
            iter: PermIter::new(
                alloc::vec![
                    ip_v4_range.a,
                    ip_v4_range.b,
                    ip_v4_range.c,
                    ip_v4_range.d,
                ],
                buffer,
            ),
        }
    }

}

impl Iterator for IPv4RangeIter<'_> {

    type Item = IpAddr;

    fn next(&mut self) -> Option<Self::Item> {
        self.nth(0)
    }

    fn nth(&mut self, n: usize) -> Option<Self::Item> {
        self.iter.nth(n).map(|d| IpAddr::from([d[0], d[1], d[2], d[3]]))
    }

}

#[test]
fn test_ip_v4_range_iters() -> crate::Result<()> {
    let buffer = Buffer::with_capacity(4);

    {
        let iter = IPv4RangeIter::new(
            IPv4Range {
                a: Component::new(192, 192),
                b: Component::new(168, 168),
                c: Component::new(0, 2),
                d: Component::new(0, 2),
            },
            &buffer,
        );
        assert_eq!(
            [
                "192.168.0.0", "192.168.1.0", "192.168.2.0",
                "192.168.0.1", "192.168.1.1", "192.168.2.1",
                "192.168.0.2", "192.168.1.2", "192.168.2.2",
            ].iter().map(|s| s.to_string()).collect::<HashSet<_>>(),
            iter.map(|i| i.to_string()).collect(),
        );
    }

    {
        let mut iter = IPv4RangeIter::new(
            IPv4Range {
                a: Component::new(0, 0),
                b: Component::new(0, 0),
                c: Component::new(0, 0),
                d: Component::new(0, 0),
            },
            &buffer,
        );
        assert_eq!(iter.next().unwrap().to_string(), "0.0.0.0");
        assert!(iter.next().is_none());
    }

    Ok(())
}
