// License: see LICENSE file at root directory of `master` branch

use {
    std::{
        fs,
        path::PathBuf,
    },

    dia_args::{
        Result,
        paths::{self, permissions},
    },
};

#[test]
fn paths() -> Result<()> {
    paths::verify_ne_file_name("123", None)?;
    paths::verify_ne_file_name("it's ok", None)?;

    paths::verify_ne_file_name("z".repeat(1024), None)?;
    paths::verify_ne_file_name("a".repeat(1025), None).unwrap_err();

    paths::verify_ne_file_name("  123   ", None).unwrap_err();
    paths::verify_ne_file_name("abc", Some(1)).unwrap_err();
    paths::verify_ne_file_name("/", None).unwrap_err();
    paths::verify_ne_file_name("\\", None).unwrap_err();
    paths::verify_ne_file_name("hello \n there", None).unwrap_err();
    paths::verify_ne_file_name("hi \t again", None).unwrap_err();

    Ok(())
}

#[test]
#[cfg(unix)]
fn file_permissions() -> Result<()> {
    let file = PathBuf::from(file!()).canonicalize()?.parent().unwrap().join(format!("{}-invalid-permissions", dia_args::DIA_ARGS_FILE_NAME));

    for mode in &[0o300_u32, 0o700, 0o610, 0o630, 0o650, 0o670, 0o601, 0o603, 0o605, 0o607] {
        use std::os::unix::fs::OpenOptionsExt;

        if file.is_file() {
            fs::remove_file(&file)?;
        }
        drop(std::fs::OpenOptions::new().write(true).create(true).mode(*mode).open(&file)?);

        permissions::ensure_safe(&file, *mode).unwrap();
        permissions::ensure_safe(&file, 0).unwrap_err();
    }

    Ok(())
}
