// License: see LICENSE file at root directory of `master` branch

use {
    std::{
        collections::HashSet,
        env,
        fs,
    },

    dia_args::{MergeOption, Result},
};

const ARG_TEST: &[&str] = &["-t", "--test"];
const ARG_DEBUG: &[&str] = &["--debug"];
const ARG_FLAG: &[&str] = &["-f", "--flag"];
const ARG_SOME: &[&str] = &["--some"];

#[test]
fn args() -> Result<()> {
    let args = dia_args::parse_strings([
        "cmd", "-b", "-fghj=FALSE", "--src=123", "/path/to/some/file", "--some", "none", "--type", "ogg", "--type", "m4v", "--type", "M4V",
        "-", "--ids", "9", "--ids", "1", "--ids", "99", "--ids", "22", "--ids", "333",
        "--", "sub", "args", "--can", "--be", "--malformed:~!@#$%^&*()_+", "-", "--", "-", "--",
    ].iter()).unwrap();
    assert_eq!(args.args().unwrap(), &["cmd", "/path/to/some/file"]);
    assert_eq!(args.use_stdin(), true);
    assert_eq!(args.options().get("-b").unwrap(), &[dia_args::TRUE_AS_STR]);
    assert_eq!(args.options().get("-f").unwrap(), &[dia_args::FALSE_AS_STR]);
    assert_eq!(args.options().get("-g").unwrap(), &[dia_args::FALSE_AS_STR]);
    assert_eq!(args.options().get("-h").unwrap(), &[dia_args::FALSE_AS_STR]);
    assert_eq!(args.options().get("-j").unwrap(), &[dia_args::FALSE_AS_STR]);
    assert_eq!(args.options().get("--src").unwrap(), &["123"]);
    assert!(args.options().get("--SRC").is_none());
    assert_eq!(args.options().get("--some").unwrap(), &["none"]);
    assert_eq!(args.options().get("--type").unwrap(), &["ogg", "m4v", "M4V"]);
    assert_eq!(args.sub_args().unwrap(), &["sub", "args", "--can", "--be", "--malformed:~!@#$%^&*()_+", "-", "--", "-", "--"]);

    assert_eq!(true, args.get(&["-b"])?.unwrap());
    args.get::<bool>(&["-b", "-f"]).unwrap_err();
    assert_eq!(123_u8, args.get(&["--src"])?.unwrap());
    args.get::<u8>(&["--src", "--ids"]).unwrap_err();
    assert!(args.get::<u8>(&["--SRC"])?.is_none());
    assert_eq!(false, args.get(&["-j"])?.unwrap());
    assert_eq!("none", args.get::<String>(ARG_SOME)?.unwrap());

    assert_eq!(&["ogg", "m4v", "M4V"], args.get_vec::<String>(&["--type"])?.unwrap().as_slice());
    assert_eq!(
        &["ogg", "m4v", "M4V", "9", "1", "99", "22", "333"],
        args.get_vec::<String>(&["--type", "--ids"])?.unwrap().as_slice()
    );
    assert_eq!(&[9, 1, 99, 22, 333], args.get_vec(&["--ids"])?.unwrap().as_slice());

    Ok(())
}

#[test]
fn sub_cmds() -> Result<()> {
    let args = dia_args::parse_strings(["help", "version", "1"].iter()).unwrap();
    let (cmd, args) = args.into_sub_cmd();
    assert_eq!(cmd.unwrap(), "help");
    assert_eq!(args.args().unwrap(), &["version", "1"]);

    let (cmd, args) = args.into_sub_cmd();
    assert_eq!(cmd.unwrap(), "version");
    assert_eq!(args.args().unwrap(), &["1"]);

    let (cmd, args) = args.into_sub_cmd();
    assert_eq!(cmd.unwrap(), "1");
    assert!(args.args().is_none());

    let (cmd, args) = args.into_sub_cmd();
    assert_eq!(cmd, None);
    assert!(args.args().is_none());

    let (cmd, args) = args.into_sub_cmd();
    assert_eq!(cmd, None);
    assert!(args.args().is_none());

    Ok(())
}

#[test]
fn default_boolean_options() -> Result<()> {
    let args = dia_args::parse_strings(["--debug"].iter()).unwrap();
    assert_eq!(true, args.get(&["--debug"])?.unwrap());

    let args = dia_args::parse_strings(["--debug", "something"].iter()).unwrap();
    assert_eq!("something", args.get::<String>(&["--debug"])?.unwrap());

    let args = dia_args::parse_strings(["--debug=false", "something"].iter()).unwrap();
    assert_eq!(false, args.get(&["--debug"])?.unwrap());

    let args = dia_args::parse_strings(["--debug=false", "--other"].iter()).unwrap();
    assert_eq!(false, args.get(&["--debug"])?.unwrap());

    let args = dia_args::parse_strings(["--debug", "-x"].iter()).unwrap();
    assert_eq!(true, args.get(&["--debug"])?.unwrap());

    Ok(())
}

#[test]
fn parse_file() -> Result<()> {
    let file = env::temp_dir().join(format!("{}{}", dia_args::ID, dia_args::DIA_ARGS_FILE_NAME));

    if file.is_file() {
        fs::remove_file(&file)?;
    }

    fs::write(
        &file,
        concat!(
            "           command            \n",
            "           sub-command            \n",
            "--passphrase=something \t secret\n",
            "--type\togg\n",
            "--type         rs              \n",
            "# Some comment here\n",
            "   #       Some comment there\n",
            "--port     =       9999            \n",
        )
    )?;

    // Test max size
    for max_size in &[0, 1, 2] {
        assert!(dia_args::parse_file(Some(&file), None, Some(*max_size)).is_err());
    }

    let (cmd, args) = dia_args::parse_file(Some(&file), None, Some(file.metadata()?.len()))?.unwrap().into_sub_cmd();
    assert_eq!(cmd.unwrap(), "command");

    let (cmd, mut args) = args.into_sub_cmd();
    assert_eq!(cmd.unwrap(), "sub-command");
    assert_eq!(args.take::<String>(&["--passphrase"])?.unwrap(), "something \t secret");
    assert_eq!(vec!["ogg", "rs"], {
        let mut vec = args.take_vec::<String>(&["--type"])?.unwrap();
        vec.sort();
        vec
    });
    assert_eq!(args.take::<u16>(&["--port"])?.unwrap(), 9999);
    assert!(args.is_empty());

    Ok(())
}

#[test]
fn parse_streams() -> Result<()> {
    let stream = b"some\0command\0--debug=true\0--some\0\0--test\0false\0\0\0\0--flag=0\0--flag\01\0--flag\02\0--\0sub\0args\0--some=none\0";

    // Test max size
    for max_size in &[0, 1, 2] {
        assert!(dia_args::parse_stream(&mut &stream[..], Some(*max_size)).is_err());
    }

    let mut args = dia_args::parse_stream(&mut &stream[..], Some(stream.len() as u64))?;
    assert_eq!(args.take_args().unwrap(), &["some", "command"]);
    assert_eq!(args.take(ARG_DEBUG)?, Some(true));
    assert_eq!(args.take(ARG_TEST)?, Some(false));
    assert!(args.take::<String>(ARG_SOME)?.unwrap().is_empty());
    assert!(args.take::<u8>(ARG_FLAG).is_err());
    assert_eq!(args.take_vec::<u8>(ARG_FLAG)?.unwrap(), &[0, 1, 2]);
    assert_eq!(args.take_sub_args().unwrap(), &["sub", "args", "--some=none"]);
    assert!(args.is_empty());

    Ok(())
}

#[test]
fn merges() -> Result<()> {
    let mut final_args = dia_args::parse_strings(["--test=true", "-f=0", "--flag=1", "--debug=true"].iter())?;

    let mut args = dia_args::parse_strings(["-t=false", "--flag=9", "--ignored=true"].iter())?;
    assert!(final_args.merge_options(&mut args, &[], MergeOption::IgnoreExisting).is_err());
    assert!(final_args.merge_options(&mut args, &[ARG_DEBUG, ARG_TEST, ARG_FLAG, &[]], MergeOption::IgnoreExisting).is_err());
    assert_eq!(final_args.merge_options(&mut args, &[ARG_TEST, ARG_FLAG], MergeOption::IgnoreExisting)?, 0);
    assert_eq!(args.options().len(), 3);
    assert_eq!(args.options().get("-t").unwrap().as_slice(), &["false"]);
    assert_eq!(args.options().get("--flag").unwrap().as_slice(), &[concat!('9')]);
    assert_eq!(args.options().get("--ignored").unwrap().as_slice(), &["true"]);
    assert_eq!(args.get(ARG_TEST)?, Some(false));
    assert_eq!(args.get(&["--ignored"])?, Some(true));
    assert_eq!(final_args.get(ARG_TEST)?, Some(true));
    assert_eq!(final_args.get(ARG_DEBUG)?, Some(true));
    assert_eq!(final_args.get_vec::<u8>(ARG_FLAG)?.unwrap().into_iter().collect::<HashSet<_>>(), vec![0, 1].into_iter().collect());

    let mut args = dia_args::parse_strings(["--test=false", "--some=none", "-d=false", "-f=3"].iter())?;
    assert_eq!(
        final_args.merge_options(&mut args, &[ARG_TEST, ARG_FLAG, ARG_SOME, &["-d"]], MergeOption::TakeAll)?,
        4,
    );
    assert!(args.options().is_empty());
    assert_eq!(final_args.get(ARG_TEST)?, Some(false));
    assert_eq!(final_args.get(ARG_DEBUG)?, Some(true));
    assert_eq!(final_args.get(&["-d"])?, Some(false));
    assert_eq!(final_args.get_vec::<u8>(ARG_FLAG)?.unwrap(), &[3]);
    assert_eq!(final_args.get::<String>(ARG_SOME)?.unwrap(), "none");

    Ok(())
}
