// License: see LICENSE file at root directory of `master` branch

//! # Kit for _Unix_ path permissions
//!
//! These features are available on non-Unix systems, but they have no effects.

use {
    std::{
        env,
        path::Path,
    },

    crate::Result,
};

#[cfg(unix)]
use {
    std::io::{Error, ErrorKind},
};

/// # Permissions (`u32`)
pub type Permissions = u32;

/// # Ensures that input file's permissions must be equal to, or _more restrictive_ than the ones you provide
///
/// This function always returns `Ok(())` on non-Unix systems.
#[cfg(unix)]
pub fn ensure_safe<P>(file: P, permissions: Permissions) -> Result<()> where P: AsRef<Path> {
    const SHIFTS: &[Permissions] = &[23, 26, 29];

    let file = file.as_ref();
    let file_permissions = {
        use std::os::unix::fs::MetadataExt;
        file.metadata()?.mode()
    };
    for bits in SHIFTS {
        let src_bits = (file_permissions << bits) >> 29;
        let target_bits = (permissions << bits) >> 29;
        if src_bits > target_bits {
            return Err(Error::new(
                ErrorKind::InvalidInput,
                format!("{:?} has more permissions than expected: {:06o} > {:06o}", file, file_permissions, permissions),
            ));
        }
    }

    Ok(())
}

/// # Ensures that input file's permissions must be equal to, or _more restrictive_ than the ones you provide
///
/// This function always returns `Ok(())` on non-Unix systems.
#[cfg(not(unix))]
pub fn ensure_safe<P>(_: P, _: Permissions) -> Result<()> where P: AsRef<Path> {
    Ok(())
}

/// # Ensures that the _second_ file's permissions must be equal to, or _more restrictive_ than the first one's
///
/// This function always returns `Ok(())` on non-Unix systems.
#[cfg(unix)]
pub fn ensure_second_is_safe<P, Q>(first: P, second: Q) -> Result<()> where P: AsRef<Path>, Q: AsRef<Path> {
    ensure_safe(second, {
        use std::os::unix::fs::MetadataExt;
        first.as_ref().metadata()?.mode()
    })
}

/// # Ensures that the _second_ file's permissions must be equal to, or _more restrictive_ than the first one's
///
/// This function always returns `Ok(())` on non-Unix systems.
#[cfg(not(unix))]
pub fn ensure_second_is_safe<P, Q>(_: P, _: Q) -> Result<()> where P: AsRef<Path>, Q: AsRef<Path> {
    Ok(())
}

/// # Ensures that the input file's permissions must be equal to, or _more restrictive_ than the program's
///
/// This function always returns `Ok(())` on non-Unix systems.
pub fn ensure_safe_against_program<P>(file: P) -> Result<()> where P: AsRef<Path> {
    ensure_second_is_safe(env::current_exe()?, file)
}

/// # Ensures that input file's permissions match the ones you provide
///
/// This function always returns `Ok(())` on non-Unix systems.
///
/// ## Examples
///
/// ```
/// use dia_args::paths::permissions;
///
/// #[cfg(unix)]
/// assert!(permissions::r#match(file!(), 0o000).is_err());
///
/// #[cfg(not(unix))]
/// assert!(permissions::r#match(file!(), 0o000).is_ok());
/// ```
#[cfg(unix)]
pub fn r#match<P>(file: P, permissions: Permissions) -> Result<()> where P: AsRef<Path> {
    const SHIFT: Permissions = 23;

    let file = file.as_ref();
    let file_permissions = {
        use std::os::unix::fs::MetadataExt;
        file.metadata()?.mode()
    };
    if file_permissions << SHIFT == permissions << SHIFT {
        Ok(())
    } else {
        Err(Error::new(ErrorKind::InvalidInput, format!(
            "{file:?} has permissions: {file_permissions:06o} -- which does not match expectation: {permissions:06o}",
            file=file, file_permissions=file_permissions, permissions=permissions,
        )))
    }
}

/// # Ensures that input file's permissions match the ones you provide
///
/// This function always returns `Ok(())` on non-Unix systems.
///
/// ## Examples
///
/// ```
/// use dia_args::paths::permissions;
///
/// #[cfg(unix)]
/// assert!(permissions::r#match(file!(), 0o000).is_err());
///
/// #[cfg(not(unix))]
/// assert!(permissions::r#match(file!(), 0o000).is_ok());
/// ```
#[cfg(not(unix))]
pub fn r#match<P>(_: P, _: Permissions) -> Result<()> where P: AsRef<Path> {
    Ok(())
}
