// License: see LICENSE file at root directory of `master` branch

//! # Option

use {
    core::fmt::{self, Display},
    std::borrow::Cow,

    super::{Cfg, I18n},
};

/// # Option
///
/// ## Examples
///
/// ```
/// use std::borrow::Cow;
/// use dia_args::docs::{self, Opt};
///
/// // All these constants are convenient while working with Args.
/// // And you can also use them for Opt.
///
/// const ARG_PORT: &[&str] = &["-p", "--port"];
/// const ARG_PORT_VALUES: &[u16] = &[0, 64009];
/// const ARG_PORT_DEFAULT: u16 = ARG_PORT_VALUES[1];
/// const ARG_PORT_DOCS: Cow<str> = Cow::Borrowed("Port for server.");
///
/// let _opt = Opt::new(
///     ARG_PORT, false,
///     Some(docs::make_cow_strings(ARG_PORT_VALUES)), Some(&ARG_PORT_DEFAULT),
///     ARG_PORT_DOCS,
/// );
/// // Here you can pass this option to Docs::new(...)
/// ```
#[derive(Clone)]
pub struct Opt<'a> {
    names: &'a [&'a str],
    required: bool,
    values: Option<Vec<Cow<'a, str>>>,
    default: Option<&'a dyn Display>,
    docs: Cow<'a, str>,
}

impl<'a> Opt<'a> {

    /// # Makes new instance
    pub fn new(names: &'a [&'a str], required: bool, values: Option<Vec<Cow<'a, str>>>, default: Option<&'a dyn Display>, docs: Cow<'a, str>)
    -> Self {
        Self {
            names,
            required,
            values,
            default,
            docs,
        }
    }

    /// # Formats self
    pub fn format(&self, cfg: &Cfg, i18n: &I18n, f: &mut fmt::Formatter) -> Result<(), fmt::Error> {
        let tab = cfg.tab_len().saturating_mul(cfg.tab_level().into());
        f.write_str(&super::format(&format!("{:?}", self.names), tab, cfg.columns()))?;

        let tab = cfg.tab_len().saturating_mul(cfg.tab_level().saturating_add(1).into());
        f.write_str(&super::format(&format!("{}: {}", i18n.required, self.required), tab, cfg.columns()))?;
        if let Some(values) = self.values.as_ref() {
            let mut tmp = String::new();
            for (i, v) in values.iter().enumerate() {
                if i > 0 {
                    tmp.push_str(", ");
                }
                tmp.push_str(&v);
            }
            f.write_str(&super::format(&format!("{}: {}", i18n.values, tmp), tab, cfg.columns()))?;
        }
        if let Some(default) = self.default {
            f.write_str(&super::format(&format!("{}: {}", i18n.default, default), tab, cfg.columns()))?;
        }
        f.write_str(super::LINE_BREAK)?;
        f.write_str(&super::format(&self.docs, tab, cfg.columns()))?;
        f.write_str(super::LINE_BREAK)?;

        Ok(())
    }

}

impl<'a> From<Opt<'a>> for Cow<'a, Opt<'a>> {

    fn from(opt: Opt<'a>) -> Self {
        Cow::Owned(opt)
    }

}

impl<'a> From<&'a Opt<'a>> for Cow<'a, Opt<'a>> {

    fn from(opt: &'a Opt<'a>) -> Self {
        Cow::Borrowed(opt)
    }

}
