// License: see LICENSE file at root directory of `master` branch

//! # Command

use {
    core::fmt,
    std::borrow::Cow,

    super::{Cfg, I18n, Opt},
};

/// # Command
///
/// ## Examples
///
/// ```
/// use std::borrow::Cow;
/// use dia_args::docs::Cmd;
///
/// // All these constants are convenient while working with Args.
/// // And you can also use them for Cmd.
///
/// const CMD_HELP: &str = "help";
/// const CMD_HELP_DOCS: Cow<str> = Cow::Borrowed("Prints help and exits.");
///
/// let _cmd = Cmd::new(CMD_HELP, CMD_HELP_DOCS, None);
/// // Here you can pass this command to Docs::new(...)
/// ```
#[derive(Clone)]
pub struct Cmd<'a> {
    name: &'a str,
    docs: Cow<'a, str>,
    options: Option<Vec<Cow<'a, Opt<'a>>>>,
}

impl<'a> Cmd<'a> {

    /// # Makes new instance
    pub const fn new(name: &'a str, docs: Cow<'a, str>, options: Option<Vec<Cow<'a, Opt<'a>>>>) -> Self {
        Self {
            name,
            docs,
            options,
        }
    }

    /// # Formats self
    pub fn format(&self, cfg: &Cfg, i18n: &I18n, f: &mut fmt::Formatter) -> Result<(), fmt::Error> {
        // Name
        let tab = cfg.tab_len().saturating_mul(cfg.tab_level().into());
        f.write_str(&super::format(self.name, tab, cfg.columns()))?;
        f.write_str(super::LINE_BREAK)?;

        // Docs
        let cfg = cfg.increment_level();
        let tab = cfg.tab_len().saturating_mul(cfg.tab_level().into());
        f.write_str(&super::format(&self.docs, tab, cfg.columns()))?;
        f.write_str(super::LINE_BREAK)?;

        // Options
        if let Some(options) = self.options.as_ref() {
            f.write_str(&super::format(i18n.options.to_uppercase(), tab, cfg.columns()))?;
            f.write_str(super::LINE_BREAK)?;

            let cfg = cfg.increment_level();
            for option in options {
                option.format(&cfg, &i18n, f)?;
            }
        }

        Ok(())
    }

}

impl<'a> From<Cmd<'a>> for Cow<'a, Cmd<'a>> {

    fn from(cmd: Cmd<'a>) -> Self {
        Cow::Owned(cmd)
    }

}

impl<'a> From<&'a Cmd<'a>> for Cow<'a, Cmd<'a>> {

    fn from(cmd: &'a Cmd<'a>) -> Self {
        Cow::Borrowed(cmd)
    }

}
