// License: see LICENSE file at root directory of `master` branch

use std::{
    fs::OpenOptions,
    io::{self, Error, ErrorKind, Read, Write},
    path::PathBuf,
    time::{UNIX_EPOCH, SystemTime},
};

#[cfg(unix)]
use std::{
    os::unix::fs::FileTypeExt,
};

// ╔═════════════════╗
// ║   IDENTIFIERS   ║
// ╚═════════════════╝

/// # Unique universally identifier of this kit
#[allow(dead_code)]
pub const UUID: &str = "96860cb5-c4c1-462a-8ca2-ee18d5a30e7d";

/// # Version
#[allow(dead_code)]
pub const VERSION: &str = "0.1.1";

/// # Release date
#[allow(dead_code)]
pub const RELEASE_DATE: (u16, u8, u8) = (2019, 6, 27);

// ╔════════════════════╗
// ║   IMPLEMENTATION   ║
// ╚════════════════════╝

/// # `/dev/urandom`
#[cfg(unix)]
pub const PATH: &str = "/dev/urandom";

/// # Generates... an error
///
/// ## Notes
///
/// Currently, this function is only supported on Unix systems.
#[cfg(not(unix))]
pub fn read() -> io::Result<[u8; 256]> {
    Err(Error::new(ErrorKind::Other, "urandom is not supported on your system"))
}

/// # Tries to read some random bytes from `/dev/urandom`
#[cfg(unix)]
pub fn read() -> io::Result<[u8; 256]> {
    match PathBuf::from(PATH).metadata().map(|m| m.file_type()) {
        Ok(file_type) => match
            file_type.is_dir() == false && file_type.is_file() == false && file_type.is_char_device() && file_type.is_socket() == false
        {
            true => match OpenOptions::new().read(true).write(true).open(PATH) {
                Ok(mut file) => {
                    let mut result = [0_u8; 256];
                    match file.read_exact(&mut result) {
                        Ok(()) => {
                            // Be nice, send something back
                            let bytes = match SystemTime::now().duration_since(UNIX_EPOCH) {
                                Ok(d) => format!(
                                    "{}-{}-{}-{}", d.as_secs(), d.subsec_millis(), d.subsec_micros(), d.subsec_nanos()
                                )
                                    .into_bytes(),
                                Err(_) => UUID.as_bytes().to_vec(),
                            };
                            match file.write_all(&bytes) {
                                Ok(()) => if cfg!(test) {
                                    println!("[{}] Sent some bytes to {:?}", UUID, PATH);
                                },
                                Err(err) => eprintln!("[{}] Couldn't write to {:?} -> {}", UUID, PATH, &err),
                            };
                            Ok(result)
                        },
                        Err(_) => Err(Error::new(ErrorKind::InvalidInput, format!("Failed to read {:?}", PATH))),
                    }
                },
                Err(_) => Err(Error::new(ErrorKind::InvalidInput, format!("Failed to open {:?}", PATH))),
            },
            false => Err(Error::new(ErrorKind::InvalidInput, format!("Malformed {:?}", PATH))),
        },
        Err(_) => Err(Error::new(ErrorKind::InvalidInput, format!("Couldn't get metadata of {:?}", PATH))),
    }
}

#[test]
#[cfg(unix)]
fn test() {
    assert_ne!(read().unwrap().to_vec(), read().unwrap().to_vec());
}

#[test]
#[cfg(not(unix))]
fn test() {
    assert!(read().is_err());
}
