// License: see LICENSE file at root directory of `master` branch

const CMD_HELP: &str = "help";
const CMD_HELP_DOCS: Cow<str> = Cow::Borrowed("Prints help and exits.");

const CMD_VERSION: &str = "version";
const CMD_VERSION_DOCS: Cow<str> = Cow::Borrowed("Prints version and exits.");

/// # Main
fn main() -> Result<()> {
    if let Err(err) = run() {
        dia_args::lock_write_err(format!("{}\n", err))?;
        process::exit(1);
    }

    Ok(())
}

/// # Runs the program
fn run() -> Result<()> {
    let args = dia_args::parse()?;
    match args.cmd() {
        Some(CMD_HELP) => {
            ensure_args_are_empty(args.into_sub_cmd().1)?;
            print_help()
        },
        Some(CMD_VERSION) => {
            ensure_args_are_empty(args.into_sub_cmd().1)?;
            print_version()
        },
        Some(other) => Err(Error::new(ErrorKind::InvalidInput, format!("Unknown command: {:?}", other))),
        None => Err(Error::new(ErrorKind::Other, "Not implemented yet")),
    }
}

/// # Ensures arguments are empty
fn ensure_args_are_empty<A>(args: A) -> Result<()> where A: AsRef<Args> {
    let args = args.as_ref();
    if args.is_empty() {
        Ok(())
    } else {
        Err(Error::new(ErrorKind::InvalidInput, format!("Unknown arguments: {:?}", args)))
    }
}

/// # Makes version string
fn make_version_string<'a>() -> Cow<'a, str> {
    Cow::Owned(format!(
        "{name} {version} {release_date:?}",
        name=>>>crate_path<<<::NAME, version=>>>crate_path<<<::VERSION, release_date=>>>crate_path<<<::RELEASE_DATE,
    ))
}

/// # Prints version
fn print_version() -> Result<()> {
    dia_args::lock_write_out(format!("{}\n", make_version_string()))
}

/// # Prints help
fn print_help() -> Result<()> {
    use dia_args::docs::{Cmd, Docs};

    let commands = Some(dia_args::make_cmds![
        Cmd::new(CMD_HELP, CMD_HELP_DOCS, None),
        Cmd::new(CMD_VERSION, CMD_VERSION_DOCS, None),
    ]);

    let mut docs = Docs::new(make_version_string(), >>>crate_path<<<::NAME.into());
    docs.commands = commands;
    docs.print()
}
