// License: see LICENSE file at root directory of `master` branch

//! # Helper binary

#![warn(missing_docs)]

#![allow(clippy::bool_comparison)]
#![allow(clippy::cognitive_complexity)]
#![allow(clippy::match_bool)]

/// # Wrapper for format!(), which prefixes your message with: crate::TAG, module_path!(), line!()
macro_rules! __ { ($($arg: tt)+) => {
    format!("[{tag}][{module_path}-{line}] {fmt}", tag=crate::TAG, module_path=module_path!(), line=line!(), fmt=format!($($arg)+))
};}

mod template_type;
mod templates;
mod urandom;

use {
    std::{
        borrow::Cow,
        env,
        io::{Error, ErrorKind},
        path::Path,
        process,
        time::{UNIX_EPOCH, SystemTime},
    },

    dia_args::{Args, Result},
    dia_time::Time,
    template_type::TemplateType,
    zeros::{Hash, Keccak},
};

const TAG: &str = dia_args::TAG;

const CMD_HELP: &str = "help";
const CMD_HELP_DOCS: Cow<str> = Cow::Borrowed("Prints help and exits.");

const CMD_VERSION: &str = "version";
const CMD_VERSION_DOCS: Cow<str> = Cow::Borrowed("Prints version and exits.");

const CMD_MAKE_TEMPLATE: &str = "make-template";
macro_rules! cmd_make_template_doc_template { () => { concat!(
    "Makes template and prints it to stdout.\n\n",
    "This command will:\n\n",
    "- Generate some constants, such as NAME, CODE_NAME, ID(1), VERSION, RELEASE_DATE, TAG...\n",
    "- Generate some functions in case your type is \"{}\".\n\n",
    "(1) To generate crate ID, the program will collect these data:\n\n",
    "- Current time.\n",
    "- Some bytes from /dev/urandom (on Unix).\n",
    "- Current directory path and some of its sub files' path.\n",
    "- Temporary directory path and some of its sub files' path.\n",
    "- Environment variables.\n",
    "- ...\n\n",
    "Then those data will be fed to an SHA3-512 hasher. Final result will be the hash.\n",
)}}

const ARG_TEMPLATE_TYPE: &[&str] = &["--type"];
const ARG_TEMPLATE_TYPE_DOCS: Cow<str> = Cow::Borrowed("Template type.");
const ARG_TEMPLATE_TYPE_VALUES: &[TemplateType] = &[TemplateType::LIB, TemplateType::BIN];
const ARG_TEMPLATE_TYPE_DEFAULT: TemplateType = TemplateType::LIB;

const ARG_NAME: &[&str] = &["--name"];
const ARG_NAME_DOCS: Cow<str> = Cow::Borrowed("Crate name.");

const ARG_CODE_NAME: &[&str] = &["--code-name"];
macro_rules! arg_code_name_doc_template { () => { concat!(
    "Crate code name.\n\n",
    "If not provided, it will be made from {:?} option, by lowering its content and replacing white spaces with hyphens (\"-\").",
)}}

/// # Main
fn main() {
    if let Err(err) = run() {
        eprintln!("{}", err);
        process::exit(1);
    }
}

/// # Runs the program
fn run() -> Result<()> {
    let args = dia_args::parse()?;
    match args.cmd() {
        Some(CMD_HELP) => {
            ensure_args_are_empty(args.into_sub_cmd().1)?;
            print_help()
        },
        Some(CMD_VERSION) => {
            ensure_args_are_empty(args.into_sub_cmd().1)?;
            print_version()
        },
        Some(CMD_MAKE_TEMPLATE) => make_template(args.into_sub_cmd().1),
        Some(other) => Err(Error::new(ErrorKind::InvalidInput, format!("Unknown command: {:?}", other))),
        None => Err(Error::new(ErrorKind::Other, "Not implemented yet")),
    }
}

/// # Ensures arguments are empty
fn ensure_args_are_empty<A>(args: A) -> Result<()> where A: AsRef<Args> {
    let args = args.as_ref();
    if args.is_empty() {
        Ok(())
    } else {
        Err(Error::new(ErrorKind::InvalidInput, format!("Unknown arguments: {:?}", args)))
    }
}

/// # Makes version string
fn make_version_string<'a>() -> Cow<'a, str> {
    format!("{} {} {:?}", dia_args::NAME, dia_args::VERSION, dia_args::RELEASE_DATE).into()
}

/// # Prints version
fn print_version() -> Result<()> {
    println!("{}", make_version_string());
    Ok(())
}

/// # Prints help
fn print_help() -> Result<()> {
    use dia_args::docs::{self, Cmd, Docs, Opt, Project};

    let commands = Some(dia_args::make_cmds![
        Cmd::new(CMD_HELP, CMD_HELP_DOCS, None),
        Cmd::new(CMD_VERSION, CMD_VERSION_DOCS, None),
        Cmd::new(
            CMD_MAKE_TEMPLATE, format!(cmd_make_template_doc_template!(), TemplateType::BIN).into(),
            Some(dia_args::make_opts![
                Opt::new(
                    ARG_TEMPLATE_TYPE, false, Some(docs::make_cow_strings(ARG_TEMPLATE_TYPE_VALUES)), Some(&ARG_TEMPLATE_TYPE_DEFAULT),
                    ARG_TEMPLATE_TYPE_DOCS,
                ),
                Opt::new(ARG_NAME, true, None, None, ARG_NAME_DOCS),
                Opt::new(ARG_CODE_NAME, false, None, None, format!(arg_code_name_doc_template!(), ARG_NAME).into()),
            ]),
        ),
    ]);
    let project = {
        let license = include_str!("../../../LICENSE");
        Some(Project::new(
            "https://bitbucket.org/haibison/dia-args", license.lines().next().unwrap(), Some(Cow::Borrowed(license)),
        ))
    };

    let mut docs = Docs::new(make_version_string(), dia_args::NAME.into());
    docs.commands = commands;
    docs.project = project;
    docs.print()?;

    Ok(())
}

/// # Makes template
fn make_template(mut args: Args) -> Result<()> {
    use templates::{
        CRATE_NAME_PLACEHOLDER, CRATE_CODE_NAME_PLACEHOLDER, CRATE_ID_PLACEHOLDER, TAG_ID_PLACEHOLDER,
        YEAR_PLACEHOLDER, MONTH_PLACEHOLDER, DAY_PLACEHOLDER,
    };

    let r#type = args.take(ARG_TEMPLATE_TYPE)?.unwrap_or(ARG_TEMPLATE_TYPE_DEFAULT);

    let name = args.take::<String>(ARG_NAME)?.ok_or_else(|| Error::new(ErrorKind::InvalidInput, format!("Missing {:?}", ARG_NAME)))?;
    if name.is_empty() {
        return Err(Error::new(ErrorKind::InvalidInput, format!("{:?} is empty", ARG_NAME)));
    }

    let code_name = match args.take::<String>(ARG_CODE_NAME)? {
        Some(code_name) => code_name.trim().to_string(),
        None => name.to_lowercase().trim().split_whitespace().collect::<Vec<_>>().join(concat!('-')),
    };
    if code_name.is_empty() {
        return Err(Error::new(ErrorKind::InvalidInput, format!("{:?} is empty", ARG_CODE_NAME)));
    }

    ensure_args_are_empty(args)?;

    let (id, prefix) = generate_crate_id()?;
    let time = Time::make_local()?;
    println!("{}", templates::header());
    if r#type == TemplateType::BIN {
        println!("{}", templates::uses());
    }
    println!(
        "{}",
        templates::identifiers()
            .replace(CRATE_NAME_PLACEHOLDER, &name).replace(CRATE_CODE_NAME_PLACEHOLDER, &code_name)
            .replace(CRATE_ID_PLACEHOLDER, &id).replace(TAG_ID_PLACEHOLDER, &prefix)
            .replace(YEAR_PLACEHOLDER, &time.year().to_string())
            .replace(MONTH_PLACEHOLDER, &time.month().order().to_string())
            .replace(DAY_PLACEHOLDER, &time.day().to_string())
    );
    if r#type == TemplateType::BIN {
        println!("{}", templates::footer());
    }

    Ok(())
}

/// # Generates crate ID and its prefix
fn generate_crate_id() -> Result<(String, String)> {
    const PREFIX_LEN: usize = 8;
    macro_rules! tab { () => { concat!(' ', ' ', ' ', ' ') }}

    let hex = {
        let mut keccak = Hash::Sha3_512.new_keccak();

        // Time
        let time = match SystemTime::now().duration_since(UNIX_EPOCH) {
            Ok(d) => d,
            Err(e) => e.duration(),
        };
        for n in &[time.as_secs() as u128, time.as_millis(), time.as_micros(), time.as_nanos()] {
            keccak.update(&n.to_ne_bytes());
        }

        // /dev/urandom
        if let Some(bytes) = read_urandom_if_available()? {
            keccak.update(bytes);
        }

        // Some paths
        for path in &[env::current_dir(), env::current_exe(), Ok(env::temp_dir())] {
            if let Ok(path) = path {
                keccak.update(path.display().to_string());
                if path.is_dir() {
                    collect_hashes_from_file_paths(path, &mut keccak);
                }
            }
        }

        // Environment variables
        for (k, v) in env::vars_os() {
            for s in &[k, v] {
                if let Some(s) = s.to_str() {
                    keccak.update(s);
                }
            }
        }

        keccak.finish_as_hex()
    };

    let hex_len = hex.len();
    if hex_len > PREFIX_LEN && hex_len / 2 == 64 {
        let prefix = hex[..PREFIX_LEN].to_string();

        let mut id = String::with_capacity(hex_len.saturating_mul(2));
        id += concat!("concat!(\n", tab!(), '"');
        for (i, c) in hex.chars().enumerate() {
            id.push(c);

            let k = i + 1;
            if k != hex_len && k % 8 == 0 {
                id.push('-');
            }

            if k % 64 == 0 {
                id += concat!('"', ',', '\n');
                id += if k < hex_len {
                    concat!(tab!(), '"')
                } else {
                    concat!(')')
                };
            }
        }

        Ok((id, prefix))
    } else {
        Err(Error::new(ErrorKind::Other, __!("Internal error")))
    }
}

/// # Reads /dev/urandom if available
#[cfg(not(unix))]
fn read_urandom_if_available() -> Result<Option<[u8; 256]>> {
    Ok(None)
}

/// # Reads /dev/urandom if available
#[cfg(unix)]
fn read_urandom_if_available() -> Result<Option<[u8; 256]>> {
    urandom::read().map(|bytes| Some(bytes))
}

/// # Finds some files in given directory and feeds their paths to given keccak
fn collect_hashes_from_file_paths<P>(dir: P, keccak: &mut Keccak) where P: AsRef<Path> {
    const MAX_FILES: usize = 1_000;

    let file_discovery = match dia_files::find_files(dir, true, dia_files::filter::AllPaths::new()) {
        Ok(file_discovery) => file_discovery,
        Err(_) => return,
    };
    for (i, file) in file_discovery.enumerate() {
        if let Ok(file) = file {
            keccak.update(file.display().to_string());
        }
        if i >= MAX_FILES {
            return;
        }
    }
}
