// License: see LICENSE file at root directory of `master` branch

//! # Arguments

mod stream;

use {
    core::{
        fmt::Debug,
        str::FromStr,
    },
    std::{
        collections::HashMap,
        env,
        fs::File,
        io::{BufReader, Error, ErrorKind, Read},
        path::Path,
    },

    crate::{
        MergeOption, Result,
        paths::permissions::{self, Permissions},
    },

    stream::*,
};

/// # `true` as string
pub const TRUE_AS_STR: &str = "true";

/// # `false` as string
pub const FALSE_AS_STR: &str = "false";

/// # Description of argument file format, in English
///
/// This constant can be useful if you want to include in your program's documentation. It has a title and its own content. So you don't have to
/// prefix with any message/description.
///
/// ## Examples
///
/// ```
/// use std::borrow::Cow;
/// use dia_args::{
///     DIA_ARGS_FILE_FORMAT, DIA_ARGS_FILE_NAME,
///     docs::{Cfg, Docs, I18n},
/// };
///
/// let docs = format!(
///     concat!(
///         // Here is program documentation.
///         "This program does something.\n\n",
///         "Options can be set via either command line or from {:?} file at root",
///         " directory of the program. Options set via command line will",
///         " override the ones from file.\n\n",
///
///         // Here is description of argument file format. Note that we don't need
///         // a title for it.
///         "{}",
///     ),
///     DIA_ARGS_FILE_NAME, DIA_ARGS_FILE_FORMAT,
/// );
/// Docs::new(Cow::Borrowed("Some Program"), Cow::Owned(docs)).print()?;
///
/// # Ok::<_, std::io::Error>(())
/// ```
pub const DIA_ARGS_FILE_FORMAT: &str = concat!(
    "Argument file format:\n\n",
    "- Empty lines or lines starting with `#` will be ignored.\n",
    "- Each command, argument, or option must be placed on a separate line.\n",
    "- Option key and value are separated by either: equal symbol `=` (can have leading/trailing white spaces), or at least one white space.",
    ' ', "Key and value will be trimmed.",
);

#[test]
fn test_dia_args_file_format() -> Result<()> {
    use crate::docs::Docs;

    Docs::new("Some Program".into(), format!("This program does something.\n\n{}", DIA_ARGS_FILE_FORMAT).into()).print()?;

    Ok(())
}

/// # Arguments
///
/// ## Notes
///
/// - [`sub_args()`][::Args::sub_args()] are **not** verified. They are the ones after `--` phrase, which (often) are meant for being passed to
///   sub processes.
///
/// [::Args::sub_args()]: struct.Args.html#method.sub_args
#[derive(Default, Debug, Clone)]
pub struct Args {

    /// # Arguments
    ///
    /// If it's some vector, the vector is _not_ empty.
    args: Option<Vec<String>>,

    /// # Raw options
    options: HashMap<String, Vec<String>>,

    /// # Sub arguments
    ///
    /// If it's some vector, the vector is _not_ empty.
    sub_args: Option<Vec<String>>,

    /// # Flag for using stdin
    use_stdin: bool,

}

impl AsRef<Args> for Args {

    fn as_ref(&self) -> &Self {
        self
    }

}

impl Args {

    /// # Gets the command
    ///
    /// This is simply the first one of [`args()`][::args()].
    ///
    /// [::args()]: #method.args
    pub fn cmd(&self) -> Option<&str> {
        self.args.as_ref().map(|args| args.first().map(|s| s.as_str())).unwrap_or(None)
    }

    /// # Gets arguments
    ///
    /// If it's some vector, the vector is _not_ empty.
    ///
    /// The first one can be used as a command, via [`cmd()`][::cmd()].
    ///
    /// [::cmd()]: #method.cmd
    pub fn args(&self) -> Option<Vec<&str>> {
        self.args.as_ref().map(|args| args.iter().map(|s| s.as_str()).collect())
    }

    /// # Gets raw options
    ///
    /// Normally you don't need this. Instead, you can use [`get()`][::get()], [`get_vec()`][::get_vec()]...
    ///
    /// [::get()]: #method.get
    /// [::get_vec()]: #method.get_vec
    pub fn options(&self) -> &HashMap<String, Vec<String>> {
        &self.options
    }

    /// # Sub arguments
    ///
    /// If it's some vector, the vector is _not_ empty.
    pub fn sub_args(&self) -> Option<Vec<&str>> {
        self.sub_args.as_ref().map(|v| v.iter().map(|s| s.as_str()).collect())
    }

    /// # Checks if there are no arguments/options/sub arguments
    pub fn is_empty(&self) -> bool {
        self.args.is_none() && self.options.is_empty() && self.sub_args.is_none()
    }

    /// # Gets flag for using stdin
    pub fn use_stdin(&self) -> bool {
        self.use_stdin
    }

    /// # Transforms into sub command
    ///
    /// For example:
    ///
    /// - Command line:
    ///     ```shell
    ///     ~> program help version 1
    ///     ```
    ///
    /// - Parsed as:
    ///     ```code
    ///     help version 1
    ///     ```
    ///
    /// - After calling this function:
    ///     ```code
    ///     version 1
    ///     ```
    ///
    /// ```
    /// use dia_args;
    ///
    /// const CMD_VERSION: &str = "version";
    ///
    /// let (cmd, args) = dia_args::parse()?.into_sub_cmd();
    /// match cmd.as_ref().map(|s| s.as_str()) {
    ///     Some(CMD_VERSION) => if args.is_empty() {
    ///         println!("Version: ...");
    ///     } else {
    ///         eprintln!("{:?} command doesn't take arguments", CMD_VERSION);
    ///     },
    ///     Some(other) => eprintln!("Command {:?} not supported", other),
    ///     None => eprintln!("Missing command"),
    /// };
    ///
    /// # Ok::<_, std::io::Error>(())
    /// ```
    pub fn into_sub_cmd(self) -> (Option<String>, Self) {
        let mut args = self.args;
        (
            match args.as_mut() {
                Some(args) => if args.is_empty() {
                    None
                } else {
                    Some(args.remove(0))
                },
                None => None,
            },
            Self {
                args: match args {
                    Some(args) => if args.is_empty() {
                        None
                    } else {
                        Some(args)
                    },
                    None => None,
                },
                ..self
            }
        )
    }

    /// # Gets a value
    ///
    /// You can provide multiple keys as you want. But the user can only provide one single key of them. This function can be used for short
    /// version and long version of your keys.
    ///
    /// ## Examples
    ///
    /// ```
    /// use dia_args;
    ///
    /// let args = dia_args::parse_strings(["--type", "ogg"].iter())?;
    /// assert_eq!(args.get::<String>(&["-t", "--type"])?.unwrap(), "ogg");
    ///
    /// let args = dia_args::parse_strings(["--type", "ogg"].iter())?;
    /// assert!(args.get::<String>(&["-t"])?.is_none());
    ///
    /// let args = dia_args::parse_strings(["--type", "ogg", "-t", "some"].iter())?;
    /// args.get::<String>(&["-t", "--type"]).unwrap_err();
    ///
    /// # Ok::<_, std::io::Error>(())
    /// ```
    pub fn get<T>(&self, keys: &[&str]) -> Result<Option<T>> where T: FromStr, <T as FromStr>::Err: Debug {
        let mut result = None;

        for key in keys {
            if let Some(values) = self.options.get(*key) {
                if result.is_none() {
                    match values.len() {
                        0 => return Err(Error::new(ErrorKind::InvalidData, __!("Internal parser error: no values found"))),
                        1 => result = Some(
                            T::from_str(&values[0]).map_err(|err|
                                Error::new(ErrorKind::InvalidData, format!("Failed parsing value {:?} of {:?}: {:?}", &values[0], &key, err))
                            )?
                        ),
                        _ => return Err(Error::new(ErrorKind::InvalidData, format!("Expected 1 value, got: {:?}", &values))),
                    };
                } else {
                    return Err(Error::new(ErrorKind::InvalidData, format!("Duplicate value for {:?}", key)));
                }
            }
        }

        Ok(result)
    }

    /// # Calls [`get()`][self::get()] and removes the keys if the result is `Ok(Some)`
    ///
    /// ## Examples
    ///
    /// ```
    /// use dia_args;
    ///
    /// let mut args = dia_args::parse_strings(["--type", "rs"].iter())?;
    /// assert_eq!(args.take::<String>(&["--type"])?.unwrap(), "rs");
    /// assert!(args.get::<String>(&["--type"])?.is_none());
    ///
    /// # Ok::<_, std::io::Error>(())
    /// ```
    ///
    /// [self::get()]: struct.Args.html#method.get
    pub fn take<T>(&mut self, keys: &[&str]) -> Result<Option<T>> where T: FromStr, <T as FromStr>::Err: Debug {
        let result = self.get(keys);
        if result.as_ref().map(|r| r.is_some()).unwrap_or(false) {
            for key in keys {
                self.options.remove(*key);
            }
        }

        result
    }

    /// # Gets a vector of values
    ///
    /// ## Examples
    ///
    /// ```
    /// use dia_args;
    ///
    /// let args = dia_args::parse_strings(
    ///     ["--type", "ogg", "-t", "m4v", "--type", "md", "-t", "rs"].iter()
    /// )?;
    /// let mut types = args.get_vec::<String>(&["-t", "--type"])?.unwrap();
    /// types.sort();
    /// assert_eq!(types, &["m4v", "md", "ogg", "rs"]);
    ///
    /// # Ok::<_, std::io::Error>(())
    /// ```
    pub fn get_vec<T>(&self, keys: &[&str]) -> Result<Option<Vec<T>>> where T: FromStr, <T as FromStr>::Err: Debug {
        let mut result: Option<Vec<_>> = None;

        for key in keys {
            if let Some(values) = self.options.get(*key) {
                for v in values.iter() {
                    let v = T::from_str(v)
                        .map_err(|err| Error::new(ErrorKind::InvalidData, format!("Failed parsing value {:?} of {:?}: {:?}", v, key, err)))?;
                    match result {
                        Some(ref mut result) => result.push(v),
                        None => result = Some(vec![v]),
                    };
                }
            }
        }

        Ok(result)
    }

    /// # Calls [`get_vec()`][self::get_vec()] and removes the keys if the result is `Ok(Some)`
    ///
    /// ## Examples
    ///
    /// ```
    /// use dia_args;
    ///
    /// let mut args = dia_args::parse_strings(["-l", "c", "-l", "c++"].iter())?;
    /// let mut languages = args.take_vec::<String>(&["-l"])?.unwrap();
    /// languages.sort();
    /// assert_eq!(languages, &["c", "c++"]);
    /// assert!(args.is_empty());
    ///
    /// # Ok::<_, std::io::Error>(())
    /// ```
    ///
    /// [self::get_vec()]: struct.Args.html#method.get_vec
    pub fn take_vec<T>(&mut self, keys: &[&str]) -> Result<Option<Vec<T>>> where T: FromStr, <T as FromStr>::Err: Debug {
        let result = self.get_vec(keys);
        if result.as_ref().map(|r| r.is_some()).unwrap_or(false) {
            for key in keys {
                self.options.remove(*key);
            }
        }

        result
    }

    /// # Takes arguments out
    ///
    /// ## Examples
    ///
    /// ```
    /// use dia_args;
    ///
    /// let mut args = dia_args::parse_strings(["do", "this", "--faster=true"].iter())?;
    /// assert_eq!(args.take_args().unwrap(), &["do", "this"]);
    ///
    /// # Ok::<_, std::io::Error>(())
    /// ```
    pub fn take_args(&mut self) -> Option<Vec<String>> {
        self.args.take()
    }

    /// # Takes sub arguments out
    ///
    /// ## Examples
    ///
    /// ```
    /// use dia_args;
    ///
    /// let mut args = dia_args::parse_strings(
    ///     ["eat", "chicken", "--", "with", "ronnie-coleman"].iter()
    /// )?;
    /// assert_eq!(args.take_sub_args().unwrap(), &["with", "ronnie-coleman"]);
    ///
    /// # Ok::<_, std::io::Error>(())
    /// ```
    pub fn take_sub_args(&mut self) -> Option<Vec<String>> {
        self.sub_args.take()
    }

    /// # Merges _options_ with other
    ///
    /// - This function works on _options_, not commands/stdin flag/sub arguments...
    /// - Other's options will be taken out, if conditions are met.
    /// - Result is number of items merged.
    ///
    /// ## Parameters
    ///
    /// - `filter`:
    ///
    ///     + If you provide some sets of keys, only those (from other) are accepted.
    ///     + If you provide an empty slice, or any of its items is empty, an error is returned.
    ///
    /// ## Examples
    ///
    /// Your program allows the user to set options from file. Later you want to give the user new ability to set options via command line,
    /// overwriting the ones from file. Then this function can help.
    ///
    /// ```
    /// use dia_args::MergeOption;
    ///
    /// const ARG_DEBUG: &[&str] = &["-d", "--debug"];
    /// const ARG_PORT: &[&str] = &["--port"];
    ///
    /// // Here in test, we're parsing from strings.
    /// // In real code, you might want to use dia_args::parse_file()
    /// let mut args_from_file = dia_args::parse_strings(
    ///     ["--debug=false", "--port=6789"].iter()
    /// )?;
    ///
    /// // Command line arguments
    /// let mut cmd_line_args = dia_args::parse_strings(
    ///     ["-d=true", "--address", "localhost"].iter()
    /// )?;
    ///
    /// // Merge
    /// let count = cmd_line_args.merge_options(
    ///     &mut args_from_file, &[ARG_DEBUG, ARG_PORT], MergeOption::IgnoreExisting,
    /// )?;
    /// assert_eq!(count, 1);
    ///
    /// // Verify
    /// assert_eq!(cmd_line_args.get(ARG_DEBUG)?, Some(true));
    /// assert_eq!(cmd_line_args.get::<String>(&["--address"])?.unwrap(), "localhost");
    /// assert_eq!(cmd_line_args.get::<u16>(ARG_PORT)?, Some(6789));
    ///
    /// # Ok::<_, std::io::Error>(())
    /// ```
    pub fn merge_options(&mut self, other: &mut Self, filter: &[&[&str]], merge_option: MergeOption) -> Result<usize> {
        if filter.is_empty() || filter.iter().any(|keys| keys.is_empty()) {
            return Err(Error::new(ErrorKind::InvalidInput, format!("Invalid filter: {:?}", filter)));
        }

        let mut count = 0;

        for (key, other_value) in other.options.drain().collect::<HashMap<_, _>>() {
            let keys = {
                let k = key.as_str();
                match filter.iter().find(|keys| keys.contains(&k)) {
                    Some(keys) => keys,
                    None => {
                        other.options.insert(key, other_value);
                        continue;
                    },
                }
            };

            match merge_option {
                MergeOption::TakeAll => keys.iter().for_each(|k| drop(self.options.remove(*k))),
                MergeOption::IgnoreExisting => if keys.iter().any(|k| self.options.contains_key(*k)) {
                    other.options.insert(key, other_value);
                    continue;
                },
            };

            self.options.insert(key, other_value);
            count += 1;
        }

        Ok(count)
    }

}

///  # Kinds of a single argument
#[derive(Debug, Eq, PartialEq)]
enum ArgKind {

    /// # A command
    Command,

    /// # A short option
    ShortOption,

    /// # A short option with value
    ShortOptionWithValue,

    /// # Multiple boolean short options
    SomeBoolShortOptions,

    /// # A long option
    LongOption,

    /// # A long option with value
    LongOptionWithValue,

    /// # Sub arguments separator
    SubArgsSeparator,

    /// # Stdin indicator
    StdinIndicator,

}

/// # Parses from an iterator of strings
pub fn parse_strings<S, I>(args: I) -> Result<Args> where S: AsRef<str>, I: Iterator<Item=S> {
    let mut result = Args {
        args: None,
        options: HashMap::new(),
        sub_args: None,
        use_stdin: false,
    };

    /// # Adds option
    fn add_option(arg: &str, value: String, mut options: Option<&mut Vec<String>>, allow_duplicates: bool) -> Result<Option<Vec<String>>> {
        if let Some(options) = options.as_mut() {
            if allow_duplicates || options.contains(&value) == false {
                options.push(value);
                return Ok(None);
            } else {
                return Err(Error::new(ErrorKind::InvalidData, format!("Duplicate value for {:?}", &arg)));
            }
        }
        Ok(Some(vec![value]))
    }

    let mut args = args.peekable();

    while let Some(arg) = args.next() {
        let arg = arg.as_ref();
        match ArgKind::parse(arg)? {
            ArgKind::Command => {
                let arg = arg.trim();
                if arg.is_empty() == false {
                    if let Some(args) = add_option(arg, arg.to_string(), result.args.as_mut(), true)? {
                        result.args = Some(args);
                    }
                }
            },
            ArgKind::SomeBoolShortOptions => {
                let parts = arg.splitn(2, '=').collect::<Vec<&str>>();
                let value = match parts.len() {
                    1 => TRUE_AS_STR.to_string(),
                    _ => match parts[1].trim().to_lowercase().as_str() {
                        self::TRUE_AS_STR => self::TRUE_AS_STR.to_string(),
                        self::FALSE_AS_STR => self::FALSE_AS_STR.to_string(),
                        _ => return Err(Error::new(ErrorKind::InvalidData, format!("Invalid some boolean short options: {:?}", &arg))),
                    },
                };
                for chr in parts[0].chars() {
                    let arg = format!("-{}", &chr);
                    if let Some(options) = add_option(&arg, value.clone(), result.options.get_mut(&arg), false)? {
                        result.options.insert(arg, options);
                    }
                }
            },
            ArgKind::ShortOption | ArgKind::LongOption => {
                let value = match args.peek().map(|s| s.as_ref().starts_with('-')) {
                    Some(true) | None => TRUE_AS_STR.to_string(),
                    Some(false) => args.next().map(|s| s.as_ref().to_string()).ok_or_else(||
                        // This shouldn't happen, but it's better than ::unwrap()
                        Error::new(ErrorKind::InvalidData, format!("Missing value for {:?}", &arg))
                    )?,
                };
                if let Some(options) = add_option(arg, value, result.options.get_mut(arg), false)? {
                    result.options.insert(arg.to_string(), options);
                }
            },
            ArgKind::ShortOptionWithValue | ArgKind::LongOptionWithValue => {
                let parts = arg.splitn(2, '=').collect::<Vec<&str>>();
                match parts.len() {
                    2 => {
                        let (arg, value) = (parts[0].to_string(), parts[1].to_string());
                        if let Some(options) = add_option(&arg, value, result.options.get_mut(&arg), false)? {
                            result.options.insert(arg, options);
                        }
                    },
                    _ => return Err(Error::new(ErrorKind::InvalidData, format!("Invalid option: {:?}", &arg))),
                };
            },
            ArgKind::SubArgsSeparator => {
                let sub_args = args.map(|some| some.as_ref().to_string()).collect::<Vec<String>>();
                if sub_args.is_empty() == false {
                    result.sub_args = Some(sub_args);
                }
                break;
            },
            ArgKind::StdinIndicator => {
                if result.use_stdin {
                    return Err(Error::new(ErrorKind::InvalidData, format!("Duplicate {:?}", &arg)));
                } else {
                    result.use_stdin = true;
                }
            },
        };
    }

    Ok(result)
}

/// # Parses from process' arguments
pub fn parse() -> Result<Args> {
    parse_strings(env::args().skip(1))
}

/// # Parses from file
///
/// ## Rules
///
/// ### Default file
///
/// - Default file is a file named [`DIA_ARGS_FILE_NAME`][::DIA_ARGS_FILE_NAME] within directory of the program. On Unix, if the program
///   is a symlink, its parent directory is used. The parent directory of the original file is ***not*** used.<sup><a id='::&1' href='#::1'>
///   `[1]`</a></sup>
/// - If `None` is given, default file will be used.
/// - If the file does not exist, `None` is returned.
///
/// ### Limits and syntax
///
/// - If `max_size` is zero, an error is returned. If `None`, [`MAX_DIA_ARGS_FILE_SIZE`][::MAX_DIA_ARGS_FILE_SIZE] will be used. If the file's
///   size is larger than provided value, an error is returned.
/// - On Unix:
///
///     + If you don't provide permissions, the file's permissions must be _equal to or more restrictive_ than the program's owners'. Or an
///       error will be returned.
///     + If you provide permissions, the file's permissions must _match_ them exactly, or an error will be returned.
///
/// - Empty lines or lines starting with `#` will be ignored.
/// - Each command, argument, or option must be placed on a separate line.
/// - Normally, a shell will remove leading/trailing marks such as `"..."` or `'...'`. ***However*** those are _not_ required in this file. So
///   you can separate options like these:
///
///     ```shell
///     --passphrase=secret passphrase with white-spaces in it
///     --passphrase        =       secret passphrase with white-spaces in it
///     --passphrase        secret passphrase with white-spaces in it
///     ```
///
///   They're all the same. Also, note that values will be trimmed.
///
/// ---
///
/// 1. <a id='::1' href='#::&1'>`^^`</a> In theory, that is the goal. However [`env::current_exe()`][r:env::current_exe()] function might
///    return the original file (not the symlink). In that case, the parent directory of the original file will be used.
///
/// [::DIA_ARGS_FILE_NAME]: constant.DIA_ARGS_FILE_NAME.html
/// [::MAX_DIA_ARGS_FILE_SIZE]: constant.MAX_DIA_ARGS_FILE_SIZE.html
/// [r:env::current_exe()]: https://doc.rust-lang.org/std/env/fn.current_exe.html
pub fn parse_file<P>(file: Option<P>, permissions: Option<Permissions>, max_size: Option<u64>) -> Result<Option<Args>> where P: AsRef<Path> {
    // NOTES:
    //
    // - If you change file format, update documentation of this function *and* DIA_ARGS_FILE_FORMAT constant.

    let max_size = max_size.unwrap_or(crate::MAX_DIA_ARGS_FILE_SIZE);
    if max_size == 0 {
        return Err(Error::new(ErrorKind::InvalidInput, "max_size must be larger than 0"));
    }

    let current_exe = env::current_exe()?;
    let file = match file.map(|f| f.as_ref().to_path_buf()) {
        Some(file) => file,
        None => match current_exe.parent() {
            Some(dir) => dir.join(crate::DIA_ARGS_FILE_NAME),
            None => return Err(Error::new(ErrorKind::NotFound, "Could not find parent directory of the program")),
        },
    };
    let file = if file.exists() {
        if file.is_file() {
            match file.canonicalize() {
                Ok(file) => file,
                Err(err) => return Err(Error::new(err.kind(), format!("Failed getting canonical path of {:?}: {:?}", file, err))),
            }
        } else {
            return Err(Error::new(ErrorKind::InvalidInput, format!("Not a file: {:?}", file)));
        }
    } else {
        return Ok(None);
    };

    // Check file size
    let file_size = file
        .metadata().map_err(|e| Error::new(ErrorKind::Other, format!("Failed getting file size of {:?}: {:?}", file, e)))?
        .len();
    if file_size > max_size {
        return Err(Error::new(ErrorKind::InvalidInput, format!("File too large: {:?} (max allowed: {})", file, max_size)));
    }

    // Verify file permissions
    match permissions {
        Some(permissions) => permissions::r#match(&file, permissions)?,
        None => permissions::ensure_second_is_safe(&current_exe, &file)?,
    };

    let mut args: Vec<String> = vec![];
    for line in read_file_to_string(file, Some(file_size))?.lines() {
        let line = line.trim();
        if line.is_empty() || line.starts_with('#') {
            continue;
        }
        if line.starts_with('-') == false || false == ['=', ' ', '\t'].iter().any(|separator| match line.find(*separator) {
            Some(idx) => {
                args.push(format!("{}={}", line[..idx].trim(), line[idx + 1..].trim()));
                true
            },
            None => false,
        }) {
            args.push(line.to_owned());
        }
    }

    parse_strings(args.into_iter()).map(|args| Some(args))
}

/// # Reads file to string with optional limit
fn read_file_to_string<P>(file: P, limit: Option<u64>) -> Result<String> where P: AsRef<Path> {
    const BUF_SIZE: usize = 8 * 1024;

    let file = file.as_ref();
    let limit = file.metadata()?.len().min(limit.unwrap_or(u64::max_value()));
    let mut reader = BufReader::new(File::open(file)?).take(limit);

    let mut buf = [0; BUF_SIZE];
    let mut data = Vec::with_capacity(limit as usize);
    loop {
        match reader.read(&mut buf)? {
            0 => return String::from_utf8(data).map_err(|e| Error::new(ErrorKind::InvalidData, e)),
            read => data.extend(&buf[..read]),
        };
    }
}

/// # Parses a stream of strings, separated by null byte (`0`)
///
/// This function can be useful for securely passing/parsing arguments across processes.
///
/// ## Notes
///
/// - If `max_size` is zero, an error is returned. If `None`, [`MAX_DIA_ARGS_FILE_SIZE`][::MAX_DIA_ARGS_FILE_SIZE] will be used. If the stream
///   has more data than provided value, an error is returned.
/// - If the stream contains an invalid UTF-8 string, an error is returned.
/// - The stream is used as-is. So you might want to use [`BufReader`][r::BufReader].
///
/// ## Examples
///
/// ```
/// let stream = b"run\0--faster=true";
///
/// let args = dia_args::parse_stream(&mut &stream[..], None)?;
/// assert_eq!(args.cmd(), Some("run"));
/// assert_eq!(args.get(&["--faster"])?, Some(true));
///
/// # Ok::<_, std::io::Error>(())
/// ```
///
/// [::MAX_DIA_ARGS_FILE_SIZE]: constant.MAX_DIA_ARGS_FILE_SIZE.html
/// [r::BufReader]: https://doc.rust-lang.org/std/io/struct.BufReader.html
pub fn parse_stream<R>(stream: &mut R, max_size: Option<u64>) -> Result<Args> where R: Read {
    let max_size = max_size.unwrap_or(crate::MAX_DIA_ARGS_FILE_SIZE);
    if max_size == 0 {
        return Err(Error::new(ErrorKind::InvalidInput, "max_size must be larger than 0"));
    }

    let mut strings = Vec::with_capacity(128);
    for s in Stream::make(stream, max_size)? {
        strings.push(s?);
    }
    parse_strings(strings.into_iter())
}

impl ArgKind {

    /// # Parses long options
    ///
    /// 2 leading characters are expected to be `--`. They won't be verified but will be ignored.
    fn parse_long_options(s: &str) -> Result<ArgKind> {
        for (index, chr) in s.chars().skip(2).enumerate() {
            match chr {
                '=' => match index {
                    0 => return Err(Error::new(ErrorKind::InvalidData, format!("Invalid long option: {:?}", &s))),
                    _ => return Ok(ArgKind::LongOptionWithValue),
                },
                'a'..='z' | 'A'..='Z' | '0'..='9' | '-' => continue,
                _ => return Err(Error::new(ErrorKind::InvalidData, format!("Invalid long option: {:?}", &s))),
            };
        }

        Ok(ArgKind::LongOption)
    }

    /// # Parses short options
    ///
    /// The first leading character is expected to be `-`. It won't be verified but will be ignored.
    fn parse_short_options(s: &str) -> Result<ArgKind> {
        let mut must_be_some_bool_options = false;
        let mut chars = s.char_indices().skip(1);
        while let Some((index, chr)) = chars.next() {
            match chr {
                '=' => match index {
                    // This shouldn't be true, we skipped 1 above
                    0 => return Err(Error::new(ErrorKind::Other, __!("Invalid index: {} -> {}", &index, &chr))),
                    1 => return Err(Error::new(ErrorKind::InvalidData, format!("Invalid short option: {:?}", &s))),
                    2 => return Ok(ArgKind::ShortOptionWithValue),
                    _ => match chars.map(|(_, chr)| chr).collect::<String>().trim().to_lowercase().as_str() {
                        self::TRUE_AS_STR | self::FALSE_AS_STR => return Ok(ArgKind::SomeBoolShortOptions),
                        _ => return Err(Error::new(ErrorKind::InvalidData, format!("Invalid some boolean short options: {:?}", &s))),
                    },
                },
                'a'..='z' | 'A'..='Z' | '0'..='9' => if index > 1 {
                    must_be_some_bool_options = true;
                },
                _ => return Err(Error::new(ErrorKind::InvalidData, format!("Invalid short option: {:?}", &s))),
            };
        }
        if must_be_some_bool_options {
            Ok(ArgKind::SomeBoolShortOptions)
        } else {
            Ok(ArgKind::ShortOption)
        }
    }

    /// # Parses argument kind
    fn parse(s: &str) -> Result<ArgKind> {
        if s.starts_with("--") {
            return match s.len() {
                2 => Ok(ArgKind::SubArgsSeparator),
                _ => Self::parse_long_options(s),
            };
        }
        if s.starts_with('-') {
            return match s.len() {
                1 => Ok(ArgKind::StdinIndicator),
                _ => Self::parse_short_options(s),
            };
        }

        Ok(ArgKind::Command)
    }

}
