[![Latest Version]][crates.io] [![Documentation]][docs.rs] ![License]

A fixed size ring buffer for datagrams of various sizes.

Implementation uses `std::collections::VecDeque` for metadata storage and a fixed size backing buffer for storage of datagram data.

# Examples

Usage as store and forward buffer for UNIX datagram socket.

```rust
use dgrambuf::DatagramBuf;
use std::os::unix::net::UnixDatagram;

fn main() -> std::io::Result<()> {
    let socket = UnixDatagram::bind("/path/to/my/socket")?;

    // allocate backing buffer
    let mut dgram_buf = Vec::new();
    dgram_buf.resize(512, 0);

    let mut dgram_buf = DatagramBuf::from_slice(&mut dgram_buf);

    // receive 10 datagrams up to 128 bytes in length each
    for _ in 0..10 {
        // drop old datagrams if there is not enough space left in the backing buffer (512)
        let mut buf = dgram_buf.alloc_front_drop(128).unwrap();

        let count = socket.recv(&mut buf)?;
        // reduce the size of the allocation to fit the datagram received
        dgram_buf.truncate_front(count);
    }

    // send back the received datagrams in order
    while let Some(mut buf) = dgram_buf.pop_back() {
        socket.send(&mut buf)?;
    }
    Ok(())
}
```

[crates.io]: https://crates.io/crates/dgrambuf
[Latest Version]: https://img.shields.io/crates/v/dgrambuf.svg
[Documentation]: https://docs.rs/dgrambuf/badge.svg
[docs.rs]: https://docs.rs/dgrambuf
[License]: https://img.shields.io/crates/l/dgrambuf.svg
