// @license magnet:?xt=urn:btih:1f739d935676111cfff4b4693e3816e664797050&dn=gpl-3.0.txt GPL-v3-or-Later
// note: JS automatically adds all elements with IDs to the global namespace.

// --- SORTING ---

const sorting = {
	key: null,
	ascending: true,
};
const sorting_methods = {
	thumbnailType: (a, b) => a.localeCompare(b),
	name: (a, b) => a.localeCompare(b),
	modifiedTime: (a, b) => parseInt(a) - parseInt(b),
	size: (a, b) => parseInt(a) - parseInt(b),
};

for (const [header_name, compare_fn] of Object.entries(sorting_methods)) {
	const element = document.getElementById(`header--${header_name}`);
	const handler = (_event) => {
		if (sorting.key === header_name) {
			sorting.ascending = !sorting.ascending;
		} else {
			sorting.key = header_name;
			sorting.ascending = true;
		}
		const actual_compare_fn = sorting.ascending ? compare_fn : (a, b) => -compare_fn(a, b);
		resort_listing(header_name, actual_compare_fn);
		update_sort_headers(element);
	};
	element.addEventListener("click", handler);
	element.addEventListener("keydown", (event) => {
		if (event.key == "Enter") {
			event.preventDefault();
			handler(event);
		}
	});
}

function update_sort_headers(current_header) {
	for (const header_name of Object.keys(sorting_methods)) {
		const header = document.getElementById(`header--${header_name}`);
		const indicator = header.getElementsByClassName("sort-status")[0];
		indicator.innerText = "-";
	}
	current_header.getElementsByClassName("sort-status")[0].innerText = sorting.ascending ? "^" : "v";
}

function resort_listing(field_name, sort_fn) {
	const with_get = (field_name, sort_fn) => (a, b) => sort_fn(a.dataset[field_name], b.dataset[field_name]);
	// always use name as the tiebreaker
	const with_tiebreaker = (a, b) => with_get(field_name, sort_fn)(a, b) || with_get("name", (a, b) => a.localeCompare(b))(a, b);
	const children = [...listing.children];
	children.sort(with_tiebreaker);
	listing.replaceChildren(...children);
	preview_setup_handlers();
}

// --- PREVIEWS ---

let previewable_count = 0;
let preview_current_index = null;
let preview_current = null;

function preview_setup_handlers() {
	previewable_count = 0;
	for (const entry of listing.children) {
		const rich_preview = entry.dataset.richPreview;
		if (rich_preview) {
			const index = previewable_count;
			const link = entry.getElementsByClassName("entry-link")[0];
			link.classList.add("entry-link_rich-preview");
			link.onclick = (e) => {
				e.preventDefault();
				preview_open(entry, index);
			};
			++previewable_count;
		}
	}
}
preview_setup_handlers();
document.getElementById("preview-position-total").innerText = previewable_count;

function preview_open(entry, index) {
	preview.classList.add("open");

	preview_current_index = index;
	preview_current = entry;

	document.getElementById("preview-title").innerText = entry.dataset.name;
	document.getElementById("preview-meta").innerText = entry.getElementsByClassName("entry-size")[0].innerText;
	document.getElementById("preview-position-current").innerText = index + 1;

	document.getElementById("preview-item")?.remove();
	document.getElementById("preview-item-container").appendChild(preview_make_container_for_entry(entry.dataset.name, entry.dataset.richPreview));
}

function preview_make_container_for_entry(name, rich_type) {
	const src = window.location.href + "/" + name;
	let ret;
	switch (rich_type) {
		case "image": {
			ret = document.createElement("img");
			ret.src = src;
			ret.alt = name;
			break;
		}
		case "video": {
			ret = document.createElement("video");
			ret.src = src;
			ret.controls = true;
			ret.autoplay = true;
			break;
		}
		default:
			console.error(`Attempt to make container for non-previewable entry (rich_type is ${rich_type})`);
			return;
	}
	ret.id = "preview-item";
	return ret;
}

function preview_close() {
	preview_exit_fullscreen();
	slideshow_stop();
	document.getElementById("preview-item")?.remove();
	preview.classList.remove("open");
	preview_current = null;
}

// the `again` argument specifies whether the function will make the browser prefetch the previous/next item by moving to the previous/next item for a split second.
function preview_previous(again = true) {
	let previous = preview_current;
	const previous_index = preview_current_index - 1;
	do {
		if (previous.previousElementSibling) {
			previous = previous.previousElementSibling;
		} else {
			return false;
		}
	} while (!previous.dataset.richPreview);
	preview_open(previous, previous_index);
	if (again && preview_previous(false)) {
		setTimeout(() => preview_next(false), 10);
	}
	return true;
}

function preview_next(again = true) {
	let next = preview_current;
	const next_index = preview_current_index + 1;
	do {
		if (next.nextElementSibling) {
			next = next.nextElementSibling;
		} else {
			return false;
		}
	} while (!next.dataset.richPreview);
	preview_open(next, next_index);
	if (again && preview_next(false)) {
		setTimeout(() => preview_previous(false), 10);
	}
	return true;
}

function preview_first() {
	let first = listing.firstElementChild;
	while (!first.dataset.richPreview) {
		if (!first) {
			return false;
		}
		first = first.nextElementSibling;
	}
	preview_open(first, 0);
	return true;
}

function preview_last() {
	let last = listing.lastElementChild;
	while (!last.dataset.richPreview) {
		if (!last) {
			return false;
		}
		last = last.previousElementSibling;
	}
	preview_open(last, previewable_count - 1);
	return true;
}

document.addEventListener("keydown", (event) => {
	if (preview_current !== null) {
		switch (event.key) {
			case "Escape": {
				event.preventDefault();
				preview_close();
				break;
			}
			case "ArrowLeft": {
				event.preventDefault();
				preview_previous();
				break;
			}
			case "ArrowRight": {
				event.preventDefault();
				preview_next();
				break;
			}
			case "Home": {
				event.preventDefault();
				preview_first();
				break;
			}
			case "End": {
				event.preventDefault();
				preview_last();
				break;
			}
			case "s": {
				event.preventDefault();
				slideshow_toggle();
				break;
			}
		}
	}
});

// --- FULLSCREEN ---

function get_fullscreen_element() {
	return document.fullscreenElement || document.mozFullScreenElement || document.webkitFullscreenElement || document.msFullscreenElement;
}

function preview_exit_fullscreen() {
	if (!get_fullscreen_element()) {
		return;
	}
	if (document.exitFullscreen) {
		document.exitFullscreen();
	} else if (document.webkitExitFullscreen) {
		document.webkitExitFullscreen();
	} else if (document.msExitFullscreen) {
		document.msExitFullscreen();
	} else {
		alert("Your browser does not support the Fullscreen API.");
	}
}

function preview_enter_fullscreen() {
	if (preview.requestFullscreen) {
		preview.requestFullscreen();
	} else if (preview.webkitRequestFullscreen) {
		preview.webkitRequestFullscreen();
	} else if (preview.msRequestFullscreen) {
		preview.msRequestFullscreen();
	} else {
		alert("Your browser does not support the Fullscreen API.");
	}
}

function preview_toggle_fullscreen() {
	if (get_fullscreen_element()) {
		preview_exit_fullscreen();
	} else {
		preview_enter_fullscreen();
	}
}

// --- SLIDESHOW ---

let slideshow_interval = null;

function slideshow_start() {
	if (slideshow_interval) {
		return;
	}
	let timeout = prompt("Slideshow interval in seconds");
	if (!timeout) {
		return;
	}
	timeout = Math.round(parseFloat(timeout) * 1000);
	slideshow_interval = setInterval(() => preview_next(), timeout);
}

function slideshow_stop() {
	if (!slideshow_interval) {
		return;
	}
	clearInterval(slideshow_interval);
	slideshow_interval = null;
}

function slideshow_toggle() {
	if (slideshow_interval) {
		slideshow_stop();
	} else {
		slideshow_start();
	}
}

// @license-end
