use actix_web::http::StatusCode as HttpStatus;
use actix_web::HttpResponse;
use std::path::Path;
use std::{fs, io};

pub fn io_error_to_status(err: &io::Error) -> HttpStatus {
	match err.kind() {
		io::ErrorKind::NotFound => HttpStatus::NOT_FOUND,
		_ => HttpStatus::INTERNAL_SERVER_ERROR,
	}
}

pub fn get_file_mime(path: &Path) -> io::Result<Option<mime::Mime>> {
	use mime_sniffer::MimeTypeSnifferExt;
	let file = fs::File::open(path)?;
	let data = match unsafe { memmap::Mmap::map(&file) } {
		// if the file is empty, the MMap will fail, but we wouldn't be able to determine the MIME anyway, so just return `None`.
		Err(err) if err.kind() == io::ErrorKind::InvalidInput => {
			return Ok(None);
		}
		x => x?,
	};
	Ok(data.sniff_mime_type_ext())
}

pub trait MimeIsRich {
	fn richness(&self) -> Option<RichType>;
}

impl MimeIsRich for Option<mime::Mime> {
	fn richness(&self) -> Option<RichType> {
		self.as_ref().and_then(MimeIsRich::richness)
	}
}

impl MimeIsRich for mime::Mime {
	fn richness(&self) -> Option<RichType> {
		let ty = self.type_().as_str();
		Some(match ty {
			"image" => RichType::Image,
			"video" => RichType::Video,
			_ => return None,
		})
	}
}

#[derive(Clone, Copy)]
pub enum RichType {
	Image,
	Video,
}

impl RichType {
	pub fn as_str(self) -> &'static str {
		match self {
			Self::Image => "image",
			Self::Video => "video",
		}
	}
}

pub fn dereference_symlink(path: &Path) -> io::Result<std::path::PathBuf> {
	let new_path = fs::read_link(&path)?;
	Ok(if new_path.is_relative() {
		path
			.parent()
			.ok_or_else(|| io::Error::from(io::ErrorKind::NotFound))?
			.join(new_path)
	} else {
		new_path
	})
}

#[derive(askama::Template)]
#[template(path = "error.html")]
pub struct ErrorTemplate {
	code: u16,
	description: Option<&'static str>,
	error: String,
}

pub fn error_response<E>(error: &E) -> HttpResponse
where
	E: actix_web::ResponseError + std::error::Error,
{
	use askama::Template as _;

	let status_code = error.status_code();
	let template = ErrorTemplate {
		code: status_code.as_u16(),
		description: status_code.canonical_reason(),
		error: error.to_string(),
	};
	match template.render() {
		Ok(template) => HttpResponse::build(status_code)
			.insert_header(actix_web::http::header::ContentType(mime::TEXT_HTML))
			.body(template),
		Err(error) => HttpResponse::build(HttpStatus::INTERNAL_SERVER_ERROR).body(error.to_string()),
	}
}
