use super::error::Error;
use crate::config::Config;
use crate::paths::{FsPath, ThumbnailPath};
use std::collections::HashMap;
use std::sync::Arc;
use tokio::sync::{Notify, RwLock};

#[derive(PartialEq, Eq, Hash, Clone, Debug)]
pub struct Task {
	input_path: FsPath,
	thumbnail_size: u8,
}

impl Task {
	pub fn new(input_path: FsPath, thumbnail_size: u8) -> Task {
		Self {
			input_path,
			thumbnail_size,
		}
	}

	fn output_path(&self, config: &Config) -> ThumbnailPath {
		ThumbnailPath::from_fs_path(self.thumbnail_size, &self.input_path, config)
	}
}

pub struct TaskTracker {
	/// Arc<Notify> allows a copy of the Notify to be taken out of the HashMap so the RwLock can be released
	tasks: RwLock<HashMap<Task, Arc<Notify>>>,
	config: Arc<Config>,
}

impl TaskTracker {
	pub fn new(config: Arc<Config>) -> Arc<Self> {
		log::debug!("initialize task tracker");
		Arc::new(Self {
			tasks: RwLock::new(HashMap::new()),
			config,
		})
	}

	pub async fn create_thumbnail(&self, task: Task) -> Result<ThumbnailPath, Error> {
		log::debug!("create thumbnail: {task:?}");
		let tasks = self.tasks.read().await;
		let output_path = task.output_path(&self.config);
		// if the task already exists, just await it
		if let Some(notify) = tasks.get(&task) {
			log::debug!("task already exists for {task:?}, awaiting notifier");
			let notify = Arc::clone(notify);
			std::mem::drop(tasks); // ensure that the lock is released
			notify.notified().await;
			log::debug!("got notification for {task:?}");
		} else {
			log::debug!("task does not exist for {task:?}");
			std::mem::drop(tasks);
			log::debug!("creating and inserting notifier for {task:?}");
			let notify = Arc::new(Notify::new());
			self.tasks.write().await.insert(task.clone(), notify);
			let config = Arc::clone(&self.config);
			log::debug!("spawning blocking task for {task:?}");
			let (task, result) = tokio::task::spawn_blocking(move || {
				log::debug!("blocking task now running for {task:?}");
				let result = super::gen::generate(task.thumbnail_size, &task.input_path, &config);
				log::debug!("blocking task completed for {task:?}");
				(task, result)
			})
			.await
			.unwrap();
			let notify = self.tasks.write().await.remove(&task).unwrap();
			notify.notify_waiters();
			// do this after notifying waiters, to avoid deadlocks
			result?;
		}
		Ok(output_path)
	}
}
