use crate::config::Config;
use crate::paths::{FsPath, UserPath};
use actix_web::{get, web, Responder};
use std::str::FromStr as _;
use std::sync::Arc;

mod error;
pub use error::Error;
use error::IoErrorWithContextExt as _;

mod task_tracker;
use task_tracker::{Task, TaskTracker};

mod gen;

pub type ThumbnailSize = u8;
pub const THUMBNAIL_SIZE: ThumbnailSize = 64;

pub fn tracker(config: Arc<Config>) -> Arc<TaskTracker> {
	TaskTracker::new(config)
}

pub fn configure(app: &mut web::ServiceConfig, tracker: Arc<TaskTracker>) {
	app.app_data(web::Data::from(tracker)).service(handler);
}

#[get("/thumbnail/{thumb_size}/{path:.+}")]
pub async fn handler(
	req: web::Path<(u8, String)>,
	config: web::Data<Config>,
	tracker: web::Data<TaskTracker>,
) -> Result<impl Responder, Error> {
	let (thumb_size, raw_path) = req.into_inner();
	let user_path = UserPath::from_str(&raw_path)?;
	let fs_path = FsPath::from_user_path(&user_path, &config)?;
	let thumbnail_path = tracker
		.create_thumbnail(Task::new(fs_path, thumb_size))
		.await?;
	thumbnail_path
		.to_response()
		.await
		.io_err_ctx("serving thumbnail file")
}
