use crate::config::Config;
use crate::paths::{FsPath, ThumbnailPath};
use crate::util::get_file_mime;
use mime::Mime;
use std::fs;
use std::path::Path;

use super::error::{Error, IoErrorWithContextExt as _};

pub fn generate(thumb_size: u8, fs_path: &FsPath, config: &Config) -> Result<(), Error> {
	let thumbnail_path = ThumbnailPath::from_fs_path(thumb_size, fs_path, config);
	let mime = get_file_mime(fs_path.as_path()).io_err_ctx("getting file mime type")?;
	if let Some(mime) = mime {
		dispatch_thumbnail_creation(mime, fs_path.as_path(), &thumbnail_path, thumb_size)
	} else {
		Err(Error::NoMime)
	}
}

struct ThumbnailFactory<'a> {
	input_path: &'a Path,
	output_path: &'a Path,
	thumbnail_size: u8,
}

fn dispatch_thumbnail_creation(
	mime: Mime,
	input_path: &Path,
	output_path: &ThumbnailPath,
	thumbnail_size: u8,
) -> Result<(), Error> {
	fs::create_dir_all(output_path.parent()).io_err_ctx("creating parent directory for thumbnail")?;
	let needs_update = if output_path.as_path().exists() {
		let input_metadata = fs::metadata(input_path).io_err_ctx("getting input file metadata")?;
		let output_metadata = fs::metadata(output_path.as_path())
			.io_err_ctx("getting output file metadata (already checked if it exists)")?;
		// `duration_since` returns `Err` if the "earlier" time is actually later. An update is needed if the input time is later than the output time, so use `is_err`.
		let ret = output_metadata
			.modified()
			.io_err_ctx("getting output file mtime")?
			.duration_since(
				input_metadata
					.modified()
					.io_err_ctx("getting input file mtime")?,
			)
			.is_err();
		if ret {
			log::debug!("File is newer than its thumbnail");
		} else {
			log::debug!("Thumbnail is newer than file");
		}
		ret
	} else {
		log::debug!("Creating new thumbnail");
		true
	};
	if !needs_update {
		log::debug!("Using cached thumbnail");
		Ok(())
	} else {
		log::debug!("Dispatching thumbnail creation");
		if input_path
			.symlink_metadata()
			.io_err_ctx("checking if input file is a symlink")?
			.file_type()
			.is_symlink()
		{
			// todo: symlink output when input is a symlink
			let input_target = crate::util::dereference_symlink(input_path)
				.io_err_ctx("dereferencing input path symlink")?;
			dispatch_thumbnail_creation(mime, &input_target, output_path, thumbnail_size)
		} else {
			let factory = ThumbnailFactory {
				input_path,
				output_path: output_path.as_path(),
				thumbnail_size,
			};
			match mime.type_().as_ref() {
				"image" => factory.image(),
				"video" => factory.video(),
				_ => Err(Error::NoMime),
			}
		}
	}
}

impl ThumbnailFactory<'_> {
	fn image_impl(
		image_obj: image::DynamicImage,
		output_path: &Path,
		thumbnail_size: u32,
	) -> Result<(), image::ImageError> {
		use image::GenericImageView;
		let threshold = thumbnail_size * 3 / 2;
		let within_threshold = image_obj.height() < threshold && image_obj.width() < threshold;
		if within_threshold {
			image_obj.save_with_format(output_path, image::ImageFormat::Png)
		} else {
			image_obj
				.thumbnail(thumbnail_size, thumbnail_size)
				.save_with_format(output_path, image::ImageFormat::Png)
		}
	}

	pub fn image(self) -> Result<(), Error> {
		let image = image::open(&self.input_path)?;
		Ok(Self::image_impl(
			image,
			self.output_path,
			self.thumbnail_size.into(),
		)?)
	}
	pub fn video(self) -> Result<(), Error> {
		let input = vid2img::FileSource::new(
			self.input_path,
			(self.thumbnail_size.into(), self.thumbnail_size.into()),
		)?;
		for frame in input.into_iter() {
			if let Some(image_data) = frame? {
				let image_data = image::load_from_memory_with_format(&image_data, image::ImageFormat::Png)?;
				return Ok(Self::image_impl(
					image_data,
					self.output_path,
					self.thumbnail_size.into(),
				)?);
			}
		}
		Err(Error::NoFrames)
	}
}
