use actix_web::http::StatusCode as HttpStatus;
use actix_web::ResponseError;
use std::io::Error as IoError;

#[derive(Debug, thiserror::Error)]
pub enum Error {
	#[error("IO error while {0}: {1}")]
	Io(&'static str, #[source] IoError),
	#[error("image error: {0}")]
	Image(#[from] image::ImageError),
	#[error("vid2img capture error: {0}")]
	Vid2ImgCapture(#[from] vid2img::CaptureError),
	#[error("vid2img stream error: {0}")]
	Vid2ImgStream(#[from] vid2img::StreamError),
	#[error("mime type is not recognized so thumbnail cannot be generated")]
	NoMime,
	#[error("the video file had no image frames")]
	NoFrames,
	#[error("bad path (user): {0}")]
	UserPath(#[from] crate::paths::user::FromStrError),
	#[error("bad path (fs): {0}")]
	FsPath(#[from] crate::paths::fs::FromUserPathError),
}

pub(super) trait IoErrorWithContextExt {
	type Ok;
	fn io_err_ctx(self, context: &'static str) -> Result<Self::Ok, Error>;
}

impl<T> IoErrorWithContextExt for Result<T, IoError> {
	type Ok = T;
	fn io_err_ctx(self, context: &'static str) -> Result<T, Error> {
		self.map_err(move |err| Error::Io(context, err))
	}
}

impl ResponseError for Error {
	fn status_code(&self) -> HttpStatus {
		match self {
			Self::Io(_reason, err) => crate::util::io_error_to_status(err),
			Self::Image(err) => match err {
				image::ImageError::IoError(err) => crate::util::io_error_to_status(err),
				_ => HttpStatus::INTERNAL_SERVER_ERROR,
			},
			Self::Vid2ImgCapture(vid2img::CaptureError::IoError(err)) => {
				crate::util::io_error_to_status(err)
			}
			Self::Vid2ImgStream(err) => match err {
				vid2img::StreamError::GstError(_) => HttpStatus::INTERNAL_SERVER_ERROR,
				vid2img::StreamError::FrameCaptureError => HttpStatus::UNSUPPORTED_MEDIA_TYPE, // if a frame couldn't be captured, something about the video's format is unsupported
			},
			Self::NoMime => HttpStatus::UNSUPPORTED_MEDIA_TYPE,
			Self::NoFrames => HttpStatus::UNSUPPORTED_MEDIA_TYPE,
			Self::UserPath(_) | Self::FsPath(_) => HttpStatus::BAD_REQUEST,
		}
	}

	fn error_response(&self) -> actix_web::HttpResponse {
		crate::util::error_response(self)
	}
}
