use std::fmt::{self, Display, Formatter};
use std::str::FromStr;

/// A path that is relative (no starting slash) and represents what the user sees.
/// Empty string = root directory.
/// Guaranteed to not contain any traversals ("..").
#[derive(Debug, Clone, PartialEq, Eq, Hash)]
#[repr(transparent)]
pub struct UserPath(Box<str>);

impl UserPath {
	pub fn parent(&self) -> Option<Self> {
		self
			.0
			.rsplit_once('/')
			.map(|(parent, basename)| {
				if parent.is_empty() && !basename.is_empty() {
					"/"
				} else {
					parent
				}
				.into()
			})
			.map(Self)
	}

	pub fn to_url(&self, thumbnail_size: u8) -> String {
		format!("/thumbnail/{thumbnail_size}/{}", self)
	}

	pub fn join(&self, child: &str) -> Option<Self> {
		if Self::check_for_traversals(child) {
			None
		} else {
			let maybe_slash = if self.0.is_empty() || child.is_empty() {
				"" // avoid adding leading or trailing slash
			} else {
				"/"
			};
			Some(Self(
				format!("{}{}{}", self, maybe_slash, child).into_boxed_str(),
			))
		}
	}

	pub fn as_relative(&self) -> &str {
		&self.0
	}

	fn check_for_traversals(fragment: &str) -> bool {
		fragment.split('/').any(|component| component == "..")
	}
}

#[derive(Debug, thiserror::Error)]
pub enum FromStrError {
	#[error("path contains traversals")]
	Traversals,
}

impl FromStr for UserPath {
	type Err = FromStrError;

	fn from_str(raw: &str) -> Result<Self, FromStrError> {
		// make relative if absolute
		let raw = raw.trim_start_matches('/');
		if Self::check_for_traversals(raw) {
			Err(FromStrError::Traversals)
		} else {
			Ok(Self(raw.into()))
		}
	}
}

impl Display for UserPath {
	fn fmt(&self, formatter: &mut Formatter) -> fmt::Result {
		self.0.fmt(formatter)
	}
}
