use crate::config::Config;
use std::path::{Path, PathBuf};

/// A path within the thumbnail directory, including the size.
/// Example: `ThumbnailPath("/tmp/indexer-thumbnails/64/path/to/image.png")`
#[derive(Debug)]
#[repr(transparent)]
pub struct ThumbnailPath(PathBuf);

impl ThumbnailPath {
	pub fn from_fs_path(thumbnail_size: u8, fs_path: &super::FsPath, config: &Config) -> Self {
		Self(
			config
				.thumbnail_tmp
				.join(thumbnail_size.to_string())
				.join(fs_path.as_relative_path(config)),
		)
	}

	pub fn as_path(&self) -> &Path {
		&self.0
	}
	pub fn parent(&self) -> &Path {
		self.0.parent().unwrap()
	}

	pub async fn to_response(&self) -> std::io::Result<impl actix_web::Responder> {
		actix_files::NamedFile::open_async(self.as_path())
			.await
			.map(|response| response.set_content_type(mime::IMAGE_PNG))
	}
}
