use crate::config::Config;
use actix_web::http::StatusCode as HttpStatus;
use std::path::{Path, PathBuf};

/// A path within the fs directory.
#[derive(Clone, PartialEq, Eq, Hash, Debug)]
#[repr(transparent)]
pub struct FsPath(PathBuf);

#[derive(Debug, thiserror::Error)]
pub enum FromUserPathError {
	#[error("file or directory does not exist")]
	NotFound,
}

impl actix_web::ResponseError for FromUserPathError {
	fn status_code(&self) -> HttpStatus {
		use FromUserPathError::*;
		match self {
			NotFound => HttpStatus::NOT_FOUND,
		}
	}

	fn error_response(&self) -> actix_web::HttpResponse {
		crate::util::error_response(self)
	}
}

impl FsPath {
	pub fn from_user_path(
		user_path: &super::UserPath,
		config: &Config,
	) -> Result<Self, FromUserPathError> {
		let fs_path = config.index_root.join(Path::new(user_path.as_relative()));

		if fs_path.exists() {
			Ok(Self(fs_path))
		} else {
			Err(FromUserPathError::NotFound)
		}
	}

	pub fn as_path(&self) -> &Path {
		&self.0
	}
	pub fn as_relative_path(&self, config: &Config) -> &Path {
		self.0.strip_prefix(&config.index_root).unwrap()
	}
}
