use actix_web::{middleware as mid, web};
use bindable::BindBindableExt as _;
use std::sync::Arc;

mod config;
mod index;
mod paths;
mod root;
mod static_files;
mod thumbnail;
mod util;

fn main() -> Result<(), Error> {
	actix_web::rt::System::new().block_on(main_())
}

#[derive(Debug, thiserror::Error)]
enum Error {
	#[error("could not extract configuration: {0}")]
	Config(#[from] figment::Error),
	#[error("IO error while {0}: {1}")]
	Io(&'static str, std::io::Error),
}

async fn main_() -> Result<(), Error> {
	let config = config::load()?;
	let config = Arc::new(config);

	simple_logger::SimpleLogger::new()
		.with_level(config.log_level)
		.init()
		.unwrap();

	let tracker = thumbnail::tracker(Arc::clone(&config));

	let server = {
		let config = Arc::clone(&config);
		let factory = move || {
			actix_web::App::new()
				.app_data(web::Data::from(Arc::clone(&config)))
				.wrap(mid::Logger::default())
				.service(root::handler)
				.service(index::handler)
				.configure(|app| thumbnail::configure(app, Arc::clone(&tracker)))
				.configure(static_files::configure)
		};
		actix_web::HttpServer::new(factory)
	};
	log::info!("Listening on {}", config.address);
	server
		.bind_bindable(&config.address)
		.map_err(|err| Error::Io("binding server", err))?
		.run()
		.await
		.map_err(|err| Error::Io("running server", err))
}
