use crate::config::Config;
use crate::paths::{FsPath, UserPath};
use actix_web::{get, web};
use actix_web::{Either, Responder};
use std::sync::Arc;

mod builder;
mod entry;
mod error;
mod template;

#[get("/fs{path:.*}")]
pub async fn handler(
	path: web::Path<(String,)>,
	config: web::Data<Config>,
) -> Result<Either<impl Responder, impl Responder>, error::Error> {
	let user_path: UserPath = path.into_inner().0.parse()?;
	let fs_path = FsPath::from_user_path(&user_path, &config)?;
	log::debug!("index handler: {user_path:?} => {fs_path:?}");
	let config = (*config).clone();
	let res = {
		let fs_path = fs_path.clone();
		let user_path = user_path.clone();
		let config = Arc::clone(&config);
		tokio::task::spawn_blocking(move || builder::Builder::new(fs_path, user_path, &config).build())
			.await
			.unwrap()
			.map(Either::Left)
	};
	if let Err(error::Error::IndexFile(_)) = res {
		actix_files::NamedFile::open_async(fs_path.as_path())
			.await
			.map(Either::Right)
			.map_err(move |error| error::Error::Io {
				reason: "serving file",
				user_path,
				error,
			})
	} else {
		res
	}
}
