use crate::paths::UserPath;
use actix_web::http::StatusCode as HttpStatus;
use actix_web::ResponseError;
use std::io::Error as IoError;

#[derive(Debug, thiserror::Error)]
pub enum Error {
	#[error("IO error for {user_path:?} while {reason}: {error}")]
	Io {
		reason: &'static str,
		user_path: UserPath,
		#[source]
		error: IoError,
	},
	#[error("attempt to index a file ({0:?})")]
	IndexFile(UserPath),
	#[error("bad path (user): {0}")]
	UserPath(#[from] crate::paths::user::FromStrError),
	#[error("bad path (fs): {0}")]
	FsPath(#[from] crate::paths::fs::FromUserPathError),
}

impl ResponseError for Error {
	fn status_code(&self) -> HttpStatus {
		match self {
			Self::Io { error, .. } => crate::util::io_error_to_status(error),
			Self::IndexFile(_) => HttpStatus::UNSUPPORTED_MEDIA_TYPE,
			Self::UserPath(_) | Self::FsPath(_) => HttpStatus::BAD_REQUEST,
		}
	}

	fn error_response(&self) -> actix_web::HttpResponse {
		match self {
			Self::FsPath(inner) => inner.error_response(),
			_ => crate::util::error_response(self),
		}
	}
}
