use super::error::Error;
use super::template as ctx;
use crate::config::Config;
use crate::paths::UserPath;
use std::borrow::Cow;
use std::path::Path;
use std::{fs, io};

pub struct EntryDataBuilder<'a> {
	pub dir_user_path: &'a UserPath,
	pub entry: fs::DirEntry,
	pub config: &'a Config,
}

impl EntryDataBuilder<'_> {
	pub fn try_build(self) -> Result<ctx::EntryData, Error> {
		macro_rules! try_ctx {
			($val:expr, $ctx:expr) => {
				$val.map_err(|error| Error::Io {
					reason: $ctx,
					user_path: self
						.dir_user_path
						.join(&self.entry.file_name().to_string_lossy())
						.unwrap(),
					error,
				})?
			};
		}

		// use std::fs::metadata to traverse symlinks
		let metadata = try_ctx!(
			std::fs::metadata(self.entry.path()),
			"getting file metadata"
		);
		let file_type = metadata.file_type();
		let path = self.entry.path();
		let size = if file_type.is_dir() {
			use crate::config::DirectorySizeOption as D;
			match self.config.directory_size {
				D::NoSize => None,
				D::Naive => Some((metadata.len(), true)),
				D::Entries => Some((
					try_ctx!(
						num_dir_entries(&path, self.config.exclude_dotfiles),
						"counting directory entries"
					),
					false,
				)),
				D::Recursive => Some((
					try_ctx!(
						dir_size_recursive(&path, self.config.exclude_dotfiles),
						"counting size of directory"
					),
					true,
				)),
			}
		} else {
			Some((metadata.len(), true))
		};
		Ok(ctx::EntryData {
			name: self.entry.file_name().to_string_lossy().into_owned(),
			thumbnail: try_ctx!(
				Self::make_thumbnail(
					file_type,
					Cow::Borrowed(&path),
					(
						self.dir_user_path,
						&self.entry.file_name().to_string_lossy(),
					),
					crate::thumbnail::THUMBNAIL_SIZE,
				),
				"getting thumbnail data"
			),
			modified_time: try_ctx!(metadata.modified(), "getting entry mtime").into(),
			size: size.map(Self::make_size),
		})
	}
	#[inline]
	fn make_size((number, should_format): (u64, bool)) -> ctx::EntrySize {
		ctx::EntrySize {
			number,
			should_format,
		}
	}
	fn make_thumbnail(
		mut file_type: fs::FileType,
		mut path: Cow<'_, Path>,
		user_path_parts: (&UserPath, &str), // separate to avoid unnecessary allocation
		thumbnail_size: u8,
	) -> io::Result<ctx::ThumbnailType> {
		use ctx::{RichThumbnail, ThumbnailType};

		while file_type.is_symlink() {
			path = crate::util::dereference_symlink(&path)?.into();
			file_type = path.metadata()?.file_type();
		}

		Ok(if file_type.is_file() {
			use crate::util::MimeIsRich as _;
			let mime = crate::util::get_file_mime(&path)?;
			if let Some(rich_type) = mime.richness() {
				ThumbnailType::RichThumbnail(RichThumbnail {
					ty: rich_type,
					url: user_path_parts
						.0
						.join(user_path_parts.1)
						.unwrap()
						.to_url(thumbnail_size),
				})
			} else {
				ThumbnailType::File
			}
		} else if file_type.is_dir() {
			ThumbnailType::Directory
		} else {
			ThumbnailType::Unknown
		})
	}
}

fn num_dir_entries(path: &Path, exclude_dotfiles: bool) -> io::Result<u64> {
	let mut count = 0;
	for entry in fs::read_dir(path)? {
		let entry = entry?;
		if filter_entry(&entry, exclude_dotfiles) {
			count += 1;
		}
	}
	Ok(count.try_into().unwrap())
}

fn dir_size_recursive(path: &Path, exclude_dotfiles: bool) -> io::Result<u64> {
	let mut size = 0;
	for entry in fs::read_dir(path)? {
		let entry = entry?;
		if !filter_entry(&entry, exclude_dotfiles) {
			continue;
		}
		size += if entry.file_type()?.is_dir() {
			dir_size_recursive(&entry.path(), exclude_dotfiles)?
		} else {
			entry.metadata()?.len()
		};
	}
	Ok(size)
}

pub fn filter_entry(entry: &fs::DirEntry, exclude_dotfiles: bool) -> bool {
	match entry.file_name().to_str() {
		Some(s) => !exclude_dotfiles || !s.starts_with('.'),
		// if it's invalid Unicode, it will be refused later anyway
		None => false,
	}
}
