use super::error::Error;
use super::template as ctx;
use crate::config::Config;
use crate::paths::{FsPath, UserPath};
use std::fs;

pub struct Builder<'a> {
	config: &'a Config,
	fs_path: FsPath,
	user_path: UserPath,
}

impl<'a> Builder<'a> {
	pub fn new(fs_path: FsPath, user_path: UserPath, config: &'a Config) -> Self {
		Self {
			fs_path,
			config,
			user_path,
		}
	}

	pub fn build(self) -> Result<ctx::Template, Error> {
		Ok(ctx::Template {
			entries: self.check_and_collect()?,
			title: self.user_path,
		})
	}

	fn check_and_collect(&self) -> Result<Vec<ctx::EntryData>, Error> {
		let metadata = fs::metadata(self.fs_path.as_path()).map_err(|error| Error::Io {
			reason: "getting metadata",
			user_path: self.user_path.clone(),
			error,
		})?;
		if metadata.is_dir() {
			self.collect_entries()
		} else {
			Err(Error::IndexFile(self.user_path.clone()))
		}
	}

	fn collect_entries(&self) -> Result<Vec<ctx::EntryData>, Error> {
		let mut ret = fs::read_dir(self.fs_path.as_path())
			.map_err(|error| Error::Io {
				reason: "reading directory contents",
				user_path: self.user_path.clone(),
				error,
			})?
			.filter(|entry| {
				entry
					.as_ref()
					.map(|entry| super::entry::filter_entry(entry, self.config.exclude_dotfiles))
					.unwrap_or(false)
			})
			.map(|entry| {
				entry
					.map_err(|error| Error::Io {
						reason: "reading directory entry",
						user_path: self.user_path.join("???").unwrap(),
						error,
					})
					.and_then(|entry| {
						super::entry::EntryDataBuilder {
							entry,
							config: self.config,
							dir_user_path: &self.user_path,
						}
						.try_build()
					})
			})
			.collect::<Result<Vec<_>, Error>>()?;
		ret.sort_unstable_by(|a, b| a.name.cmp(&b.name)); // can't use by_key due to lifetime issues
		Ok(ret)
	}
}
