use bindable::BindableAddr;
use log::LevelFilter;
use serde::Deserialize;
use std::path::PathBuf;

#[derive(Deserialize)]
pub struct Config {
	pub address: BindableAddr,
	pub index_root: PathBuf,
	pub thumbnail_tmp: PathBuf,
	#[serde(default)]
	pub directory_size: DirectorySizeOption,
	#[serde(default = "default_exclude_dotfiles")]
	pub exclude_dotfiles: bool,
	#[serde(
		default = "default_log_level",
		deserialize_with = "deserialize_level_filter"
	)]
	pub log_level: LevelFilter,
}

const fn default_log_level() -> LevelFilter {
	LevelFilter::Info
}

fn deserialize_level_filter<'de, D: serde::Deserializer<'de>>(
	de: D,
) -> Result<LevelFilter, D::Error>
where
	D::Error: serde::de::Error,
{
	let val = <std::borrow::Cow<'_, str>>::deserialize(de)?;
	Ok(match &*val {
		"trace" => LevelFilter::Trace,
		"debug" => LevelFilter::Debug,
		"info" => LevelFilter::Info,
		"warn" => LevelFilter::Warn,
		"error" => LevelFilter::Error,
		"off" => LevelFilter::Off,
		_ => {
			return Err(serde::de::Error::custom(
				"log level must be one of: trace, debug, info, warn, error, off",
			))
		}
	})
}

const fn default_exclude_dotfiles() -> bool {
	true
}

#[derive(Deserialize, Clone, Copy, PartialEq, Eq)]
#[serde(rename_all = "snake_case")]
pub enum DirectorySizeOption {
	/// Don't show the size of directories at all
	NoSize,
	/// Show the size of directories as the number of entries (subdirectories/files)
	Entries,
	/// Use the "size" returned by the operating system, which is not representative of the contents of the directory
	Naive,
	/// Determine the size of the directory by recursing into subdirectories. Can be very slow.
	Recursive,
}

impl Default for DirectorySizeOption {
	fn default() -> Self {
		Self::Entries
	}
}

pub fn load() -> figment::error::Result<Config> {
	use figment::providers::Format as _;

	let toml = figment::providers::Toml::file("dexr.toml");
	let env = figment::providers::Env::prefixed("DEXR_");
	figment::Figment::new().merge(toml).merge(env).extract()
}
