use handlebars::Handlebars;
use rocket::fs::{FileServer, Options as FSOptions};
use rocket::response::Redirect;
use rocket::{get, routes, Build, Rocket};
use std::sync::Arc;

mod config;
mod index;
mod static_files;
mod thumbnail;
mod util;

use config::Config;

#[rocket::main]
async fn main() -> anyhow::Result<()> {
	Ok(rocket()?.launch().await?)
}

fn rocket() -> anyhow::Result<Rocket<Build>> {
	let rocket = rocket::build();

	let config = rocket.figment().extract::<Config>()?;
	let config = Arc::new(config);

	let rocket = rocket
		.manage(config.clone())
		.manage({
			let mut hbs = Handlebars::new();
			hbs.set_strict_mode(true);
			hbs.register_template_string("base", std::include_str!("../templates/base.hbs"))?;
			hbs.register_template_string("index.html", std::include_str!("../templates/index.html.hbs"))?;
			hbs
		})
		.mount("/static", routes![static_files::css, static_files::js])
		.mount(
			"/fs",
			FileServer::new(&config.index_root, {
				let exclude_dotfiles = if config.exclude_dotfiles { FSOptions::None } else { FSOptions::DotFiles };
				FSOptions::NormalizeDirs | exclude_dotfiles
			}),
		)
		.mount("/fs", routes![index::index_wrapper])
		.mount("/thumbnail", routes![thumbnail::thumbnail_wrapper])
		.mount("/", routes![root]);
	Ok(rocket)
}

#[get("/")]
fn root() -> Redirect {
	Redirect::permanent("/fs")
}
