use serde::Serialize;
use std::borrow::Cow;
use std::cmp::{Ord, Ordering, PartialOrd};

#[derive(Serialize)]
pub struct Context {
	pub title: Option<String>,
	pub parent_path: Option<String>,
	pub entries: Vec<EntryData>,
}

#[derive(Serialize)]
pub struct EntryData {
	pub thumbnail: ThumbnailData,
	pub name: String,
	pub modified_time: Friendly<i64>,
	pub size: Option<Friendly<u64>>,
}

impl PartialEq for EntryData {
	fn eq(&self, other: &Self) -> bool {
		self.name == other.name
	}
}
impl Eq for EntryData {}

impl PartialOrd for EntryData {
	fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
		Some(self.name.cmp(&other.name))
	}
}
impl Ord for EntryData {
	fn cmp(&self, other: &Self) -> Ordering {
		self.partial_cmp(other).unwrap()
	}
}

#[derive(Serialize)]
pub struct ThumbnailData {
	#[serde(flatten)]
	pub alt: ThumbnailType,
	pub url: Cow<'static, str>,
}

#[derive(Serialize)]
#[serde(rename_all = "snake_case", tag = "alt", content = "rich_type")]
pub enum ThumbnailType {
	RichThumbnail(crate::util::RichType),
	File,
	Directory,
	Unknown,
}

#[derive(Serialize)]
pub struct Friendly<T> {
	pub number: T,
	pub friendly: String,
}
