use crate::config::Config;
use crate::util::{get_file_mime, image_error_to_status, io_error_to_status, stream_error_to_status, vid2img_error_to_status};
use mime::Mime;
use rocket::http::Status as HttpStatus;
use rocket::{get, State};
use std::fs;
use std::path::{Path, PathBuf};
use std::sync::Arc;

#[get("/<thumb_size>/<path..>")]
pub async fn thumbnail_wrapper(thumb_size: u32, path: PathBuf, config: &State<Arc<Config>>) -> Result<ThumbnailResponse, HttpStatus> {
	let config = (*config).clone();
	rocket::tokio::task::spawn_blocking(move || thumbnail(thumb_size, &path, &config)).await.unwrap()
}

fn thumbnail(thumb_size: u32, user_path: &Path, config: &Config) -> Result<ThumbnailResponse, HttpStatus> {
	let thumbnail_path = config.thumbnail_tmp.join(thumb_size.to_string()).join(user_path);
	let actual_path = config.index_root.join(user_path);
	let mime = get_file_mime(&actual_path).map_err(io_error_to_status)?;
	if let Some(mime) = mime {
		dispatch_thumbnail_creation(mime, &actual_path, &thumbnail_path, thumb_size)?;
		Ok(ThumbnailResponse(thumbnail_path))
	} else {
		Err(HttpStatus::NotFound)
	}
}

pub struct ThumbnailResponse(PathBuf);
impl<'r> rocket::response::Responder<'r, 'static> for ThumbnailResponse {
	fn respond_to(self, req: &'r rocket::Request<'_>) -> rocket::response::Result<'static> {
		let f = fs::File::open(self.0).map_err(io_error_to_status)?;
		let mut response = f.respond_to(req)?;
		response.set_header(rocket::http::ContentType::PNG);
		Ok(response)
	}
}

struct ThumbnailFactory<'a> {
	input_path: &'a Path,
	output_path: &'a Path,
	thumbnail_size: u32,
}

fn dispatch_thumbnail_creation(mime: Mime, input_path: &Path, output_path: &Path, thumbnail_size: u32) -> Result<(), HttpStatus> {
	fs::create_dir_all(output_path.parent().unwrap()).map_err(io_error_to_status)?;
	let needs_update = if output_path.exists() {
		let input_metadata = fs::metadata(input_path).map_err(io_error_to_status)?;
		let output_metadata = fs::metadata(output_path).map_err(io_error_to_status)?;
		// `duration_since` returns `Err` if the "earlier" time is actually later. An update is needed if the input time is later than the output time, so use `is_err`.
		let ret = output_metadata.modified().map_err(io_error_to_status)?.duration_since(input_metadata.modified().map_err(io_error_to_status)?).is_err();
		if ret {
			log::debug!("File is newer than its thumbnail");
		} else {
			log::debug!("Thumbnail is newer than file");
		}
		ret
	} else {
		log::debug!("Creating new thumbnail");
		true
	};
	if !needs_update {
		log::debug!("Using cached thumbnail");
		Ok(())
	} else {
		log::debug!("Dispatching thumbnail creation");
		if input_path.symlink_metadata().map_err(io_error_to_status)?.file_type().is_symlink() {
			let input_target = crate::util::dereference_symlink(input_path).map_err(io_error_to_status)?;
			dispatch_thumbnail_creation(mime, &input_target, output_path, thumbnail_size)
		} else {
			let factory = ThumbnailFactory { input_path, output_path, thumbnail_size };
			match mime.type_().as_ref() {
				"image" => factory.image(),
				"video" => factory.video(),
				_ => Err(HttpStatus::NotFound),
			}
		}
	}
}

impl ThumbnailFactory<'_> {
	fn image_impl(image_obj: image::DynamicImage, output_path: &Path, thumbnail_size: u32) -> Result<(), image::ImageError> {
		use image::GenericImageView;
		let threshold = thumbnail_size * 3 / 2;
		let within_threshold = image_obj.height() < threshold && image_obj.width() < threshold;
		if within_threshold {
			image_obj.save_with_format(output_path, image::ImageFormat::Png)
		} else {
			image_obj.thumbnail(thumbnail_size, thumbnail_size).save_with_format(output_path, image::ImageFormat::Png)
		}
	}

	pub fn image(self) -> Result<(), HttpStatus> {
		let image = image::open(&self.input_path).map_err(image_error_to_status)?;
		Self::image_impl(image, self.output_path, self.thumbnail_size).map_err(image_error_to_status)
	}
	pub fn video(self) -> Result<(), HttpStatus> {
		let input = vid2img::FileSource::new(self.input_path, (self.thumbnail_size, self.thumbnail_size)).map_err(vid2img_error_to_status)?;
		for frame in input.into_iter() {
			if let Some(image_data) = frame.map_err(stream_error_to_status)? {
				let image_data = image::load_from_memory_with_format(&image_data, image::ImageFormat::Png).map_err(image_error_to_status)?;
				return Self::image_impl(image_data, self.output_path, self.thumbnail_size).map_err(image_error_to_status);
			}
		}
		Err(HttpStatus::NotFound)
	}
}
