use super::context as ctx;
use crate::config::Config;
use crate::util::io_error_to_status;
use rocket::http::Status as HttpStatus;
use std::fs;
use std::path::{Path, PathBuf};

pub struct Builder<'a> {
	config: &'a Config,
	fs_path: PathBuf,
	user_path: String,
}

impl<'a> Builder<'a> {
	pub fn new(user_path: &Path, config: &'a Config) -> Self {
		Self {
			fs_path: config.index_root.join(user_path),
			config,
			user_path: format!("/{}", user_path.display()),
		}
	}
	pub fn build(self) -> Result<ctx::Context, HttpStatus> {
		Ok(ctx::Context {
			entries: self.check_and_collect()?,
			parent_path: self.user_path.rsplit_once('/').map(|(parent, basename)| if parent.is_empty() && !basename.is_empty() { "/" } else { parent }.to_owned()),
			title: Some(self.user_path),
		})
	}
	fn check_and_collect(&self) -> Result<Vec<ctx::EntryData>, HttpStatus> {
		let metadata = fs::metadata(&self.fs_path).map_err(io_error_to_status)?;
		if metadata.is_dir() {
			self.collect_entries()
		} else {
			log::warn!("Attempt to index a file ({:?})", self.user_path);
			Err(HttpStatus::InternalServerError)
		}
	}
	fn collect_entries(&self) -> Result<Vec<ctx::EntryData>, HttpStatus> {
		let mut ret: Vec<_> = fs::read_dir(&self.fs_path)
			.map_err(io_error_to_status)?
			.filter_map(some_or_warn)
			.filter(|entry| super::entry::filter_entry(entry, self.config.exclude_dotfiles))
			.filter_map(|entry| {
				some_or_warn(
					super::entry::EntryDataBuilder {
						entry,
						config: self.config,
						dir_user_path: &self.user_path,
					}
					.try_build(),
				)
			})
			.collect();
		ret.sort_unstable();
		Ok(ret)
	}
}

fn some_or_warn<T, E: std::fmt::Debug>(val: Result<T, E>) -> Option<T> {
	match val {
		Ok(val) => Some(val),
		Err(err) => {
			log::warn!("Error while indexing directory: {:?}", err);
			None
		}
	}
}
